#!/bin/bash
#
##########################################################
#
#  Script: condition_disks.sh v1.00 - (C) 2008 JanW
#
#  Conditions a set of disks and assembles them into
#  a new Metshovi diskpack. 
#
#  The script supports _only_ block device names /dev/* 
#  i.e. not /dev/disk/by-uuid/ or others.
#
##########################################################

if [ "$#" -lt "2" ]; then
	echo "Usage:  $ sudo ./condition_disks.sh /dev/sdc /dev/sdd ..."
	echo "Completely erases the specified list of disks and"
	echo "builds one main 'xfs' storage and one 'ext2' info"
	echo "partition on them."
	exit
fi

if [ "$USER" != "root" ]; then
	echo "condition_disks.sh must be run with 'sudo' or as 'root'!"
	exit
fi

echo "About to completely erase everything on the following $# disks:"
echo "------------------------------------------------------------"
for drv in $@; do
	if [ -b ${drv} ]; then
		file ${drv}
		drvname=${drv#/dev/}
		cat /proc/partitions | grep ${drvname}
		cat /etc/mtab | grep ${drvname}
		if [ -e /proc/mdstat ]; then cat /proc/mdstat | grep ${drvname}; fi
	fi
done
echo "------------------------------------------------------------"
echo "Confirm there is no important information on the disks!"

# http://ubuntuforums.org/showthread.php?t=758976&page=2
echo "Erase disks (y/n)?"
read -r REPLY
[ "$REPLY" = "y" ] || exit

##########################################################

aptitude install xfsprogs
/etc/init.d/mdadm stop

drvlist=""
raidpartlist=""
drvcnt=0
echo "Partition table backup, erase and reformat..."
for drv in $@; do
    echo "Clearing drive ${drv}"
	if [ -b ${drv} ]; then
                drvname=${drv#/dev/}
                drvlist="${drvlist} /dev/${drvname}"
                drvcnt=$(($drvcnt + 1))

                # backup MBR/partition table
                sfdisk -d /dev/${drvname} >> ${drvname}-partition-sectors.save
                dd if=/dev/zero bs=512 count=1 of=/dev/${drvname}

                # kill existing MD superblock first time, otherwise
                # mdadm may auto-restart the raid0 after the below sfdisk step
                mdadm --zero-superblock /dev/${drvname}
                mdadm --zero-superblock /dev/${drvname}1

                # partition the disk for RAID (0xfd linux raid auto-detect)
                echo ",,FD" | sfdisk /dev/${drvname} -uM
                raidpartlist="${raidpartlist} /dev/${drvname}1"

                #  blank out any older MD superblock on the new partition
                mdadm --zero-superblock /dev/${drvname}1

                sync
	fi
done
echo "List of $drvcnt erased drives: ${drvlist}"

if [ "$drvcnt" -lt 2 ]; then
    echo "Not enough drives for a RAID"
    /etc/init.d/mdadm start
    exit
fi

##########################################################


MDNR=0
if [ -e /dev/md_d0 ]; then
	MDNR=1
fi
if [ -e /dev/md_d1 ]; then
	MDNR=2
fi
if [ -e /dev/md_d2 ]; then
	MDNR=3
fi

echo "Creating a 2-partition partitionable RAID0 /dev/md_d${MDNR} from ${raidpartlist} ..."
yes | mdadm --create /dev/md_d${MDNR} --auto=mdp2 --metadata=1.2 --chunk=1024 --raid-disks=$drvcnt --level=0 ${raidpartlist}

/etc/init.d/mdadm start

echo "Creating a small info partition followed by the main data partition ..."

# normal MBR does not go beyond 2TB partitions, we need GPT instead
# echo -e ",64\n;\n" | sfdisk /dev/md_d${MDNR}

# create a GUID Partition Table (GPT) instead of MBR
parted /dev/md_d${MDNR} --script print
parted /dev/md_d${MDNR} --script mklabel gpt
parted /dev/md_d${MDNR} --script mkpart primary1 0 128M
yes | parted /dev/md_d${MDNR} -- mkpart primary2 128M -1
parted /dev/md_d${MDNR} --script print

# format the partitions "optimally"
mkfs.ext2 -F /dev/md_d${MDNR}p1
# mkfs.xfs -f -d su=1024k,sw=$drvcnt -r rtdev=/dev/md_d${MDNR}p3 /dev/md_d${MDNR}p2 
mkfs.xfs -f -d su=1024k,sw=$drvcnt /dev/md_d${MDNR}p2

##########################################################

echo "Generating /mnt/diskpack/ directories ..."

for i in `seq 0 3`; do
	mkdir -p /mnt/diskpacks/pack${i}meta
	mkdir -p /mnt/diskpacks/pack${i}data
done

echo "Mounting new diskpack to /mnt/diskpack/pack${MDNR}meta and /mnt/diskpack/pack${MDNR}data ... "
mount /dev/md_d${MDNR}p1 /mnt/diskpacks/pack${MDNR}meta
mount /dev/md_d${MDNR}p2 /mnt/diskpacks/pack${MDNR}data
chown oper:oper /mnt/diskpacks/pack${MDNR}meta
chown oper:oper /mnt/diskpacks/pack${MDNR}data
