----------------------------------------------------------------------------------
-- Company: MRO
-- Engineer: Jan Wagner
-- License: GNU GPL v3
--
-- Create Date:    12:42:12 12/03/2008
-- Design Name:
-- Module Name:    vdif_packetizer_v13 - Behavioral
-- Project Name:
-- Target Devices:
-- Tool versions:
-- Description:    Packetizes user data into a VDIF style of data frame. The resulting
--                 data frame can be directly forwarded into the 10G UDP packetizer core.
--
--                 The VDIF header is based on the "VDIF Specification Rev A.5.3" draft.
--
--                 The VDIF Extended User Data field is currently left unused (all 0).
--
--                 The external user IP must take care to align the frame generation
--                 i.e. the first write onto an integer second boundary, if desired.
--                 The Frame# within second VDIF field is maintained inside this vhdl and
--                 is reset to 0 when the Seconds input changes. If the change happens
--                 in the middle of a frame, the frame is nevertheless continued until
--                 the end and the Frame# of the next frame will be 0.
--
--                 When the 'data_in' input rate is slower than the system clock,
--                 the framing works in non-data-replacement mode. If the rates are
--                 the same the first 4 data words are overwritten with the header.
--
-- Dependencies:
--
-- Revision:
-- Additional Comments:
--
----------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

--library UNISIM;
--use UNISIM.VComponents.all;

entity vdif_packetizer_v13 is
   generic (
      frameLenLimit : integer := 1024;
      frameCountLimit : integer := 67108864
   );
   port (
      clk : in  std_logic;
      ce  : in  std_logic;
      rst : in  std_logic;

      -- Control
      enable : in std_logic;

      -- User Data input
      data_in    : in std_logic_vector (63 downto 0);
      data_valid : in std_logic;

      -- VDIF Header Field inputs (informative)
      epseconds     : in std_logic_vector (29 downto 0);  -- Seconds from the start of reference epoch
      epoch         : in std_logic_vector (5 downto 0);   -- Epoch# : epoch = 1/1/2000 + (6 months * Epoch#)
      bitspersample : in std_logic_vector (4 downto 0);   -- 0 = 1-bit, ..., 31 = 32-bit
      log2channels  : in std_logic_vector (4 downto 0);   -- 0 = 1 channel, up to 2^5=32 channels
      thread_id     : in std_logic_vector (9 downto 0);   -- thread ID from 0 to 1023
      station_id    : in std_logic_vector (15 downto 0);  -- 2-character station ID like 'Mh' or 'Jv'
      dw64_per_frame: in std_logic_vector (9 downto 0);   -- how many doublewords including the header

-- debug out
      dbg_dw64counter: out std_logic_vector (9 downto 0);
      dbg_fifo_empty: out std_logic;

      -- Output to 10G UDP packetizer
      tx_data         : out std_logic_vector (63 downto 0) := (others=>'0');
      tx_valid        : out std_logic := '0';
      tx_end_of_frame : out std_logic := '0';
      tx_discard      : out std_logic := '0'
   );
end vdif_packetizer_v13;

architecture Behavioral of vdif_packetizer_v13 is

   component simpleFIFO is
      generic (
         width : integer := 64;
         depth : integer := 8
      );
      port (
         clk   : in  std_logic;
         ce    : in  std_logic;
         rst   : in  std_logic;
         wr_en : in std_logic;
         rd_en : in std_logic;
         din   : in  std_logic_vector(width-1 downto 0);
         dout  : out std_logic_vector(width-1 downto 0);
         full  : out std_logic;
         empty : out std_logic;
         valid : out std_logic;
         debug : out std_logic_vector(31 downto 0)
      );
   end component;
   constant width : integer := 64;
   constant depth : integer := 10;

   signal datafifo_rden  :   std_logic := '0';
   signal datafifo_empty :   std_logic := '0';
   signal datafifo_valid :   std_logic := '0';
   signal datafifo_dout  :   std_logic_vector (63 downto 0);

   -- Constants
   constant HEADER_DW_LENGTH: integer := 4; -- VDIF Header length in doublewords
   type vdif_fsm_state is (
      WAIT_DATA,
      SEND_HDR_DWORD0,
      SEND_HDR_DWORD1,
      SEND_HDR_DWORD2,
      SEND_HDR_DWORD3,
      SEND_USERDATA
   );

   signal vdif_state : vdif_fsm_state := WAIT_DATA;

   signal dw64_count_prefinal : std_logic; -- '1' when one value short of a full frame
   signal dw64_count_ended : std_logic;
   signal clockin_enable   : std_logic;

   -- VDIF data frame# within current second
   signal frame_nr     : std_logic_vector(25 downto 0) := (others=>'0');

   -- 1PPS recovery from RTC / epoch seconds
   signal pps_pulse        : std_logic := '0';
   
begin

   -- User input port
   datainput_fifo: simpleFIFO
   generic map (
      width => width,
      depth => depth)
   port map (
      clk   => clk,
      ce    => '1',
      rst   => rst,
      din   => data_in,
      wr_en => data_valid,
      rd_en => datafifo_rden,
      dout  => datafifo_dout,
      empty => datafifo_empty,
      full  => open,
      valid => datafifo_valid,
      debug => open
   );

   dbg_fifo_empty <= datafifo_empty;
   
   -- put something to the 10G
   with vdif_state select tx_data <=
      -- no data
      (others => '0') when WAIT_DATA,
      -- Header dw#0: Invalid, dontcare, Epoch Secs, Epoch, Frame# within second
      ('0' & '0' & epseconds & epoch & frame_nr) when SEND_HDR_DWORD0,
      -- Header dw#1: Version, ..., not legacy
      ("000" & bitspersample & "00000000000000" 
       & dw64_per_frame & '0' & log2channels 
       & thread_id & station_id) when SEND_HDR_DWORD1,
      -- Header dw#2: Extended Data Version (0=unused), Extended user data 1+2
      (others=>'0') when SEND_HDR_DWORD2,
      -- Header dw#3: Extendend User Data
      (others=>'0') when SEND_HDR_DWORD3,
      -- VDIF Payload
      datafifo_dout when SEND_USERDATA,
      -- illegal state
      (others => '1') when others;

   with vdif_state select tx_valid <=
      '0' when WAIT_DATA,
      '1' when SEND_HDR_DWORD0,
      '1' when SEND_HDR_DWORD1,
      '1' when SEND_HDR_DWORD2,
      '1' when SEND_HDR_DWORD3,
      datafifo_valid when SEND_USERDATA,
      '0' when others;

   with vdif_state select tx_end_of_frame <=
      (dw64_count_prefinal and datafifo_valid) when SEND_USERDATA,
      '0' when others;

   -- State machine for creating VDIF frames
   frame_fsm_proc: process(clk, rst)
   begin
      if (rst = '1') then
         clockin_enable <= '0';
         tx_discard     <= '1';
         vdif_state     <= WAIT_DATA;
      else
         if rising_edge(clk) then
            -- FSM
            case vdif_state is
               when WAIT_DATA =>
                  tx_discard <= '0';
                  if datafifo_empty = '1' then
                     vdif_state <= WAIT_DATA;
                  else
                     vdif_state <= SEND_HDR_DWORD0;
                  end if;
               when SEND_HDR_DWORD0 =>
                  vdif_state <= SEND_HDR_DWORD1;
               when SEND_HDR_DWORD1 =>
                  vdif_state <= SEND_HDR_DWORD2;
               when SEND_HDR_DWORD2 =>
                  vdif_state <= SEND_HDR_DWORD3;
                  clockin_enable <= '1'; -- set 2 clks early due to latency of FIFO
               when SEND_HDR_DWORD3 =>
                  -- clockin_enable <= '1'; -- set early due to latency of FIFO
                  vdif_state <= SEND_USERDATA;
               when SEND_USERDATA =>
                  if dw64_count_ended = '1' then
                     clockin_enable <= '0';
                     if datafifo_empty = '1' then
                        vdif_state <= WAIT_DATA;
                     else
                        vdif_state <= SEND_HDR_DWORD0;
                     end if;
                  else
                     clockin_enable <= '1';
                     vdif_state <= SEND_USERDATA;
                  end if;
               when others =>
                  vdif_state <= WAIT_DATA;
            end case;
         end if;
      end if;
   end process frame_fsm_proc;

   -- Frame counter: is reset to 0 when second changes, 
   --                and is incremented after current VDIF header sent
   frame_counter_proc: process(clk, rst)
      variable framecounter_v : integer range 0 to frameCountLimit;
      variable old_epsecond_v : std_logic;
   begin
      if (rst = '1') then
         framecounter_v := 0;
         old_epsecond_v := epseconds(0);
      else
         if rising_edge(clk) then
             if ((dw64_count_prefinal and datafifo_valid) = '1') then
                 if (epseconds(0) /= old_epsecond_v) then
                    framecounter_v := 0;
                 else
                    framecounter_v := framecounter_v + 1;
                 end if;
                 old_epsecond_v := epseconds(0);
             end if;
          end if;
      end if;
      frame_nr <= std_logic_vector(to_unsigned(framecounter_v, frame_nr'length));
      dbg_dw64counter <= std_logic_vector(to_unsigned(framecounter_v, dbg_dw64counter'length));
   end process frame_counter_proc;

   -- Clock out a number of doublewords into 'vdif_userdata'
   frame_doubleword_proc: process(clk, rst)
      variable dw64_target_v : integer range 0 to frameLenLimit;
      variable dw64_count_v  : integer range 0 to frameLenLimit;
      variable dw64_count_prefinal_v : std_logic;
      variable dw64_count_done_v : std_logic;
   begin
      if rst = '1' then
         dw64_target_v := 256;
         dw64_count_v  := HEADER_DW_LENGTH;
         dw64_count_prefinal_v := '0';
         dw64_count_done_v := '0';
         datafifo_rden <= '0';
      else
         if rising_edge(clk) then
            if (clockin_enable = '1') then
               -- get more data from FIFO
               datafifo_rden <= not datafifo_empty;
               if (datafifo_valid = '1') then
                  dw64_count_v := dw64_count_v + 1;
               end if;
               -- one value short of a full frame?
               if (dw64_count_v = dw64_target_v-1) then
                  dw64_count_prefinal_v := '1';
               end if;
               -- full frame done?
               if (dw64_count_v = dw64_target_v) then
                  dw64_count_done_v := '1';
               else
                  dw64_count_done_v := '0';
               end if;
            else
               -- not assembling a frame, pre-load the configuration
               datafifo_rden <= '0';
               dw64_count_v      := HEADER_DW_LENGTH;
               dw64_target_v     := to_integer(unsigned(dw64_per_frame));
               dw64_count_prefinal_v := '0';
               dw64_count_done_v     := '0';
            end if; -- clockin_enable
         end if; -- clk
      end if; -- rst
      dw64_count_prefinal <= dw64_count_prefinal_v;
      dw64_count_ended <= dw64_count_done_v;
      -- dbg_dw64counter  <= std_logic_vector(to_unsigned(dw64_count_v, dbg_dw64counter'length));
   end process frame_doubleword_proc;

end Behavioral;
