
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

-------------------------------------------------------------------------------

entity vdifTest is

end vdifTest;

-------------------------------------------------------------------------------

architecture behavioral of vdifTest is

  component vdif_packetizer_v15 is
     generic (
        frameLenLimit : integer := 1024;
        frameCountLimit : integer := 67108864
     );
     port (
        clk : in  std_logic;
        ce  : in  std_logic;
        rst : in  std_logic;
        enable : in std_logic;
        data_in    : in std_logic_vector (63 downto 0);
        data_valid : in std_logic;
        epseconds     : in std_logic_vector (29 downto 0);  -- Seconds from the start of reference epoch
        epoch         : in std_logic_vector (5 downto 0);   -- Epoch# : epoch = 1/1/2000 + (6 months * Epoch#)
        bitspersample : in std_logic_vector (4 downto 0);   -- 0 = 1-bit, ..., 31 = 32-bit
        log2channels  : in std_logic_vector (4 downto 0);   -- 0 = 1 channel, up to 2^5=32 channels
        complex       : in std_logic; -- 1 if payload consists of complex data samples
        thread_id     : in std_logic_vector (9 downto 0);   -- thread ID from 0 to 1023
        station_id    : in std_logic_vector (15 downto 0);  -- 2-character station ID like 'Mh' or 'Jv'
        dw64_per_frame: in std_logic_vector (9 downto 0);   -- how many doublewords including the header
        dbg_dw64counter: out std_logic_vector (9 downto 0);
        dbg_fifo_empty : out std_logic;
        tx_data         : out std_logic_vector (63 downto 0) := (others=>'0');
        tx_valid        : out std_logic := '0';
        tx_end_of_frame : out std_logic := '0';
        tx_discard      : out std_logic := '0'
     );
  end component;

  -- component generics
  constant frameLenLimit : integer := 1024;
  constant frameCountLimit : integer := 67108864;
  constant width : integer := 64;

  -- component ports
  signal din_wr    : std_logic;
  signal din       : std_logic_vector(width-1 downto 0);


  signal dbg_dw64count: std_logic_vector (9 downto 0);
  signal vdif_dout    : std_logic_vector(width-1 downto 0);
  signal vdif_valid   : std_logic;
  signal vdif_eof     : std_logic;
  signal vdif_discard : std_logic;

  signal seconds : std_logic_vector (29 downto 0);
  signal epoch   : std_logic_vector (5 downto 0);
  signal bitspersample : std_logic_vector (4 downto 0);
  signal log2channels  : std_logic_vector (4 downto 0);
  signal thread_id     : std_logic_vector (9 downto 0);
  signal station_id    : std_logic_vector (15 downto 0);
  signal dw64_per_frame: std_logic_vector (9 downto 0);
  signal dbg_fifo_empty: std_logic;
  
  -- clock/reset
  signal clk : std_logic := '1';
  signal rst : std_logic := '1';

begin

  -- component instantiation
  DUT: vdif_packetizer_v15
    generic map (
      frameLenLimit => frameLenLimit,
      frameCountLimit => frameCountLimit)
    port map (
        -- inputs
        clk => clk,
        ce => '1',
        enable => '1',
        rst => rst,
        data_in => din,
        data_valid => din_wr,
        epseconds => seconds,
        epoch => epoch,
        bitspersample => bitspersample,
        log2channels => log2channels,
        complex => '1',
        thread_id => thread_id,
        station_id => station_id,
        dw64_per_frame => dw64_per_frame,
        -- dbg out
        dbg_dw64counter => dbg_dw64count,
        dbg_fifo_empty => dbg_fifo_empty,
        -- outputs
        tx_data => vdif_dout,
        tx_valid => vdif_valid,
        tx_end_of_frame => vdif_eof,
        tx_discard => vdif_discard
    );

  -- clock generation
  clock_gen: process(clk, rst)  -- clock generator and one shot clear signal
  begin
    if rst ='1' then -- happens only once
      rst <= '0' after 200 ps;
    elsif rst='0' then     -- avoid time zero glitch
      clk <= not clk after 5 ns;  -- 10 ns period
    end if;
  end process clock_gen;

  -- waveform generation
  wavegen_proc: process
  begin
    -- insert signal assignments here
    din <= (others=>'0');
    dw64_per_frame <= (others=>'0');
    din_wr <= '0';

    -- configure the VDIF headers inputs
    bitspersample  <= std_logic_vector(to_unsigned(8, bitspersample'length));
    log2channels   <= std_logic_vector(to_unsigned(0, log2channels'length));
    thread_id      <= std_logic_vector(to_unsigned(0, thread_id'length));
    station_id     <= std_logic_vector(to_unsigned(6669, station_id'length));
    dw64_per_frame <= std_logic_vector(to_unsigned(18, dw64_per_frame'length));
    epoch          <= std_logic_vector(to_unsigned(4, epoch'length));
    seconds        <= std_logic_vector(to_unsigned(0, seconds'length));

    wait until clk = '1';
    wait until clk = '1';

    -- feed it with some data
    for i in 1 to 400 loop
      wait until rising_edge(clk);
      if i <= 125 then
           seconds <= std_logic_vector(to_unsigned(1, seconds'length));
      else
           if i <= 2*125 then
               seconds <= std_logic_vector(to_unsigned(2, seconds'length));
           else
               seconds <= std_logic_vector(to_unsigned(3, seconds'length));
           end if;
      end if;
      din <= std_logic_vector(to_unsigned(i, din'length));
      din_wr <= '1';
      wait until rising_edge(clk);
      din_wr <= '0';
      wait until rising_edge(clk);
      -- wait until rising_edge(clk);
      -- wait until rising_edge(clk);
    end loop;

    assert false report "end of test" severity note;
    wait;

  end process wavegen_proc;

end behavioral;
