/*************************************************
 *                                               *
 * Simple Network UDP data to RAID receiver      *
 * Specially for you Sergei                      *
 *                                               *
 * Captures data generated by iBob S/C Firmware. * 
 * The packets are always 8008 bytes long.       *
 * The port is 46220.                            *
 * Packet headers: 32-bit Unix secs, 32-bit PSN  *
 *                                               *
 * Usage: sergei2raid seconds outputpfile        *
 *                                               *
 *************************************************/

//
// Compile
//   gcc -Wall sergei2raid.c -o sergei2raid -D_LARGEFILE64_SOURCE=1
//

#define  VER_STR "sergei2raid Version 1.0    (C) 2009 Jan Wagner"

#define _GNU_SOURCE 1
#define _LARGEFILE64_SOURCE 1  /* Large File Support (LFS) '*64()' functions. */
#define _FILE_OFFSET_BITS 64   /* Automatic '*()' --> '*64()' replacement. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <fcntl.h>
#include <unistd.h>

#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <sched.h>
#include <pthread.h>

#include <arpa/inet.h>
#include <netdb.h>
#include <sys/socket.h>

#include <signal.h> 
sig_atomic_t ctrlC_pressed = 0;

#define EXIT_OK           0
#define EXIT_FILE_ERROR   1
#define EXIT_NET_ERROR    2
#define EXIT_THREAD_ERROR 3

#define MIN(a,b) (a<b) ? a : b
typedef unsigned long long ull_t;

// -------------- GENERIC HELPERS

void SIGINT_handler(int signum) {
   ctrlC_pressed = 1;
}

void print_usage() {
     printf(" Usage: sergei2raid seconds outputfile\n\n");
}

// -------------- MAIN, UDP RECEIVER

int main(int argc, char* argv[]) {

  char     packetbuffer[16384] __attribute__ ((aligned (128)));;
  int      socket_fd;
  int      file_fd;
  int      sockopt;

  u_int32_t base_second = 0;
  u_int32_t old_second;
  u_int32_t second = 0;
  u_int32_t psn = 0;
  u_int32_t packetcounter = 0;

  volatile u_int32_t* header_word0_ptr = &packetbuffer[0];
  volatile u_int32_t* header_word1_ptr = &packetbuffer[4];

  int done = 0;

  struct addrinfo  hints;
  struct addrinfo *info;
  struct addrinfo *info_save;
  char             aibuffer[10];
  int              status;

  const int udpport = 46220;
  const int packetsize = 8008;

  struct sigaction sa;

  /* infos */
  fprintf(stderr, "\n %s\n\n", VER_STR);
  if (argc < 3) {
     print_usage();
     return EXIT_OK;
  }

  /* get arguments */
  u_int32_t duration = atoi(argv[1]);
  char* filename = argv[2];

  /* open the file */
  file_fd = open(filename, O_RDWR|O_CREAT/*|O_TRUNC*/, 
                 S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH);
  if (-1 == file_fd) {
     fprintf(stderr, "File open failed: '%s'\n", strerror(errno)); 
     return EXIT_FILE_ERROR;
  }

  /* bind UDP socket */
  memset(&hints, 0, sizeof(hints));
  hints.ai_flags    = AI_PASSIVE;
  hints.ai_family   = AF_INET;
  hints.ai_socktype = SOCK_DGRAM;

  sprintf(aibuffer, "%u", udpport);
  getaddrinfo(NULL, aibuffer, &hints, &info);

  info_save = info; status = 0;
  do {
     socket_fd = socket(info->ai_family, info->ai_socktype, info->ai_protocol);
     if (socket_fd < 0) { continue; }
     if (!bind(socket_fd, info->ai_addr, info->ai_addrlen)) {
         fprintf(stderr, "Receiving UDP on port %u\n", udpport);
         status = 1;
         break;
     }
  } while ((info = info->ai_next) != NULL);
  freeaddrinfo(info_save);

  if (0 == status) {
     fprintf(stderr, "Couldn't get a socket for port %u\n", udpport);
     return EXIT_NET_ERROR;
  }

  /* set receive buffer size */
  sockopt = 16*1024*1024;
  setsockopt(socket_fd, SOL_SOCKET, SO_RCVBUF, &sockopt, sizeof(sockopt));

  /* register Control-C / SIGINT handler */
  sigemptyset (&sa.sa_mask);
  sa.sa_flags = 0;
  sa.sa_handler = SIGINT_handler;
  sigaction (SIGINT, &sa, 0);

  /* do it */
  while (!done && !ctrlC_pressed) {

      ssize_t nwritten;

      /* read */
      size_t nread = recv(socket_fd, packetbuffer, packetsize, 0);
      if (nread != (size_t)packetsize) {
          fprintf(stderr, "Warning: received UDP of size %d != %d\n", (int)nread, packetsize);
          continue;
      }

      /* sync to "1PPS" i.e. seconds change */
      second = ntohl(*header_word0_ptr);
      psn = ntohl(*header_word1_ptr);
      if (base_second == 0) {
          if (psn > 1) { 
              continue; 
          } else { 
              fprintf(stderr, "Capture started at data second %lu frame 0\n", (unsigned long)base_second);
              base_second = second;
          }
      }

      /* write or ready! */
      if (second > (base_second + duration - 1)) {
          done = 1;
          fprintf(stderr, "\nRecording done!\n");
      } else {
          nwritten = write(file_fd, packetbuffer, packetsize);
          if (nwritten != packetsize) {
              fprintf(stderr, "short write: %d != %d\n", (int)nwritten, packetsize);
          } 
      }

      /* print second ticks */
      if (old_second != second) {
          old_second = second;
          fputc('.', stderr);
      }
  }

  if (!done) {
      fprintf(stderr, "\nCaught Control-C: last second was %d\n", second);
  }

  close(file_fd);
  return 0;
}
