/***************************************************
 **                                               **
 ** A "library" for processing simple VDIF data.  **
 ** Implements VDIF Release 1.0 6th March 2009    **
 **                                               **
 ** (C) 2009 Jan Wagner, Metsahovi Radio Obs      **
 **                                               **
 ** Licencing: GNU GPL v3                         **
 **                                               **
 ***************************************************/

#include "libvdif.h"
#include <arpa/inet.h>
#include <string.h>

///////////////////////////////////////////////////
// API Functions
///////////////////////////////////////////////////

typedef unsigned long long ull_t;

void print_qword_bits(FILE* f, const u_int64_t* data)
{
   int i, j;
   unsigned char* byte = (unsigned char*)data;
   printf("   ");
   for (i = 0; i < 8; i++, byte++) {
      printf("%02u:", i);
      for (j = 7; j >= 0; j--) {
         if (*byte & (1<<j)) { putc('1', f); }
         else { putc('0', f); }
      }
      putc(' ', f);
   }
   putc('\n', f);
}

u_int64_t htonll(u_int64_t value)
{
    static int necessary = -1;
    if (necessary == -1)
        necessary = (5 != htons(5));
    if (necessary)
        return (((u_int64_t) htonl(value & 0x00000000ffffffffLL)) << 32) | ((u_int64_t) htonl(value >> 32));
    else
        return value;
}
u_int64_t ntohll(u_int64_t value)
{
    return htonll(value);
}

void vdif_ntoh(vdifheader_t* vdif)
{
   vdif->raw.header_qw0 = ntohll(vdif->raw.header_qw0);
   vdif->raw.header_qw1 = ntohll(vdif->raw.header_qw1);
}

void print_vdif(FILE* f, vdifheader_t* vdif)
{
   const char* base = (const char*)vdif;

   vdifheader_parsed_t h;
   vdif_parse_header(base, &h);

   fprintf(f, "VDIF:\n");
   fprintf(f, "   invalid       : %u\n"
              "   legacy        : %u\n"
              "   epoch seconds : %u 0x%08X\n"
              "   epoch         : %u 0x%02X\n"
              "   frame number  : %u 0x%08X\n",
              h.invalid,
              h.legacy,
              h.seconds, h.seconds,
              h.epoch, h.epoch,
              h.frame, h.frame
   );
   fprintf(f, "   version       : %u\n"
              "   bits/sample   : %u\n"
              "   complex       : %u\n"
              "   frame qwords  : %u = %lu bytes\n"
              "   payload bytes : %u bytes\n"
              "   log2(channels): %u = %u channel\n"
              "   thread        : %u 0x%04X\n"
              "   station       : 0x%02X-0x%02X '%c%c'\n",
              h.version,
              h.bitspersample, h.complex,
              h.framelength_w64s, 8UL*(h.framelength_w64s),
              h.payloadlength_bytes,
              h.log2channels, (1 << h.log2channels),
              h.threadID, h.threadID,
              h.stationID[0], h.stationID[1],
              h.stationID[0], h.stationID[1]
   );
              
   return;
}


///////////////////////////////////////////////////
// VDIF Header Field Acessors
///////////////////////////////////////////////////

/*
 * Get individual fields from a VDIF Data Frame Header
 */

#define VDIF_WORD32(x) ((u_int32_t)ntohl(*(((const u_int32_t*)base) + x)))

char vdif_get_invalid(const char* base)
{
   return ((VDIF_WORD32(0) & (1<<31)) != 0);
}

char vdif_get_legacy(const char* base)
{
   return ((VDIF_WORD32(0) & (1<<30)) != 0);
}

char vdif_get_complex(const char* base)
{
   return ((VDIF_WORD32(3) & (1<<31)) != 0);
}

char vdif_get_version(const char* base)
{
   return (VDIF_WORD32(2) >> 29) & 3;
}

char vdif_get_log2channels(const char* base)
{
   return (VDIF_WORD32(2) >> 24) & 31;
}

char vdif_get_bitspersample(const char* base)
{
   return 1 + ((VDIF_WORD32(3) >> 26) & 31);
}

u_int32_t vdif_get_framelength(const char* base)
{
   return 1 + (VDIF_WORD32(2) & 0x00FFFFFF);
}

u_int32_t vdif_get_seconds(const char* base)
{
   return (VDIF_WORD32(0) & 1073741823ULL); // 2^30-1=1073741823
}

u_int32_t vdif_get_epoch(const char* base)
{
   return (VDIF_WORD32(1) >> 24) & 63;
}

u_int32_t vdif_get_framenumber(const char* base)
{
   return (VDIF_WORD32(1) & 0x00FFFFFF);
}

u_int32_t vdif_get_thread(const char* base)
{
   return (VDIF_WORD32(3)>>16) & 0x3F;
}

const char* vdif_get_station(const char* base)
{
   return base + 3*4 + 2;
}

const char* vdif_get_payload(const char* base)
{
   if (vdif_get_legacy(base)) {
      return base + 4*4;
   } else {
      return base + 8*4;
   }   
}

int vdif_get_extdata_version(const char* base)
{
   if (!vdif_get_legacy(base)) {
      return (VDIF_WORD32(4) >> 24) & 255;
   }
   return 0;
}

///////////////////////////////////////////////////
// VDIF Header Field Decoders
///////////////////////////////////////////////////


/*
 * Parse all VDIF Data Frame Header fields into a more
 * readily usable struct that requires no additional
 * bit fiddling.
 */
vdifheader_parsed_t* vdif_parse_header(const char* base, vdifheader_parsed_t* header)
{
   const char *c;
   if (header==NULL) return NULL;

   memset(header, 0, sizeof(vdifheader_parsed_t));

   header->base = base;
   header->payload = vdif_get_payload(base);

   header->version = vdif_get_version(base);
   header->epoch = vdif_get_epoch(base);
   header->seconds = vdif_get_seconds(base);
   header->frame = vdif_get_framenumber(base);
   header->framelength_w64s = vdif_get_framelength(base);
   header->threadID = vdif_get_thread(base);
   c = vdif_get_station(base);
   header->stationID[0] = c[0];
   header->stationID[1] = c[1];
   header->stationID[2] = '\0';

   header->invalid = vdif_get_invalid(base);
   header->legacy = vdif_get_legacy(base);
   header->complex = vdif_get_complex(base);
   header->bitspersample = vdif_get_bitspersample(base);
   header->log2channels = vdif_get_log2channels(base);

   if (!header->legacy) {
        int i;
        header->extdataversion = vdif_get_extdata_version(base);
        for (i=0; i<15; i++) {
           header->extdata[i] = *(base + 16 + 1 + i);
        }
        header->payloadlength_bytes = 8 * (header->framelength_w64s - 4);
    } else {
        header->payloadlength_bytes = 8 * (header->framelength_w64s - 2);
    }

    return header;
}
