/***************************************************
 **                                               **
 ** iBobControl                                   **
 **                                               **
 ** A "library" for sending commands to the iBob  **
 ** and waiting for a response, with timeout.     **
 **                                               **
 **************************************************/

#include "ibobcontrol.h"
	
// Open the control API
int ibobcontrol_open(ibobcontrol_t* cfg)
{
   int sotmp = 1;
   if (cfg == NULL) return -1;

   /* Bind UDP socket */
   cfg->socket_fd = socket(AF_INET, SOCK_DGRAM, 0);
   if (cfg->socket_fd < 0) {
      fprintf(stderr, "Could not create a socket: '%s'\n", strerror(errno)); 
      return -1;
   }  

   /* Disable sender UDP checksums */
   if (setsockopt(cfg->socket_fd, SOL_SOCKET, SO_NO_CHECK, &sotmp, sizeof(sotmp)) < 0) {
      fprintf(stderr, "Warning: failed to disable UDP checksums: '%s'\n", strerror(errno)); 
   }

   /* Prepare the transmit/receive infos */
   FD_ZERO( &cfg->read_set );
   FD_SET ( cfg->socket_fd, &cfg->read_set ); 
   cfg->client_addr.sin_family = AF_INET;
   cfg->client_addr.sin_port   = htons(IBOB_CTRL_PORT);
   cfg->read_timeout.tv_sec    = IBOB_TIMEOUT_SEC;
   cfg->read_timeout.tv_usec   = 0;

   return 0;
}

// Send a request and wait for the reply
int ibobcontrol_query(const char* ibob_ip, ibobcontrol_t* cfg, 
                      ibobquery_t* q, ibobresponse_t* r)
{ 
   char datagram[IBOB_MAX_DGRAM_SIZE];
   size_t len;
   int rc;
   if (cfg == NULL || q == NULL || r == NULL) return -1;
   if (ibob_ip == NULL) return -1;

   /* Empty reply */
   r->length = 0;

   /* Prepare and send the datagram */
   cfg->client_addr.sin_addr.s_addr = inet_addr(ibob_ip);   
   *(((u_int32_t*)datagram) + 0) = htonl(q->command);
   *(((u_int32_t*)datagram) + 1) = htonl(q->address);
   *(((u_int32_t*)datagram) + 2) = htonl(q->value);
   rc = sendto( cfg->socket_fd, datagram, 3*sizeof(u_int32_t), 0, 
                (struct sockaddr*)&cfg->client_addr, 
                sizeof(cfg->client_addr) );
   if (rc < 0) {
      fprintf(stderr, "sendto() failed!\n");
      return -1;
   }
   fprintf(stderr, "Command sent.\n");

   /* Wait for response */
   rc = select(cfg->socket_fd + 1, &cfg->read_set, NULL, NULL, &cfg->read_timeout);
   if (rc <= 0) {
      fprintf(stderr, "Timed out without receiving any reply.\n");
      return 0;
   }

   /* Receive the data into the result struct */
   len = recv(cfg->socket_fd, r->data, 3 * sizeof(u_int32_t), 0);
   if (len < 0) {
      fprintf(stderr, "recv() failed!\n");
      return -1;
   }
   r->length = len;
   return (int)len;
}

// Close
int ibobcontrol_close(ibobcontrol_t* cfg)
{
   close(cfg->socket_fd);
   return 0;
}
