// ss.cc -- TCP/IP socker server
// $Id: ss.cc,v 1.18 2008/07/03 09:26:46 lindi Exp $
// Copyright (C) 2000 Ari Mujunen. (Ari.Mujunen@hut.fi)

// This is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License.
// See the file 'COPYING' for details.

// $Log: ss.cc,v $
// Revision 1.18  2008/07/03 09:26:46  lindi
// Add proper receive buffers for each connection. Previously every
// command had to be sent in the same packet. This code has been running
// on dammer for months and seems to be stable. Requires 40 * 512 = 20
// kilobytes of extra RAM.
//
// Revision 1.17  2008-06-09 07:18:23  jha
// ported to the new pci416 device driver interface (not thoroughly checked for 64-bit compatibility though)
//
// Revision 1.16  2007-08-15 11:19:17  lindi
// Remove dead code (#if 0's and stuff) and comments related to them to
// make the it easier to see what is really happening in the program.
//
// Revision 1.15  2007/08/15 11:07:01  lindi
// Remove ip address check that was unused already
//
// Revision 1.14  2007/08/15 11:04:57  lindi
// _Only_ M-x indent-region with cc-mode indentation
//
// Revision 1.13  2007/01/18 11:02:25  lindi
// Reduce select timeout from 500ms to 10ms with network sockets to speed up dammer.
//
// Revision 1.12  2006/09/15 12:36:26  lindi
// common: use SO_REUSEADDR so that tcp port can be reused immediately.
//
// Revision 1.11  2005/11/30 10:14:27  lindi
// If read() from network returns multiple lines parse them separately
//
// Revision 1.10  2004/05/10 07:16:17  amn
// Removed IP address check.
//
// Revision 1.9  2002/12/18 07:31:25  amn
// Upped revision.
//
// Revision 1.2  2002/12/18 07:31:23  amn
// intermediate
//
// Revision 1.1  2002/12/18 07:23:46  amn
// Recovered from /home disk crash by re-creating 'common' module in CVS.
//
// Revision 1.8  2000/02/23 13:08:22  amn
// 'write()' error return conditions were just the opposite.
//
// Revision 1.7  2000/02/23 13:01:07  amn
// Asserts away from "UNknown command" network replies.
//
// Revision 1.6  2000/02/23 12:09:53  amn
// Added ignoring all SIGPIPE signals during 'serve_socket()' loop.
// Read/write calls will return EPIPE errors instead.
//
// Revision 1.5  2000/02/22 04:40:27  amn
// Removed 'exit()'s from the inside of serve_socket loop.
//
// Revision 1.4  2000/02/16 11:59:37  amn
// Removed connect/disconnect messages (stderr).
//
// Revision 1.3  2000/02/09 07:36:47  amn
// In case of default "unknown keyword" reply, send back a terminating '\n'.
//
// Revision 1.2  2000/02/04 13:55:49  amn
// Added networking include files.
//
// Revision 1.1  2000/01/17 14:26:50  amn
// Initial version.
//

#include "registration.h"
RCSID(ss_cc, "$Id: ss.cc,v 1.18 2008/07/03 09:26:46 lindi Exp $");


// Socket call includes.
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>  // for inet_ntoa()

#include <signal.h>  // for ignoring SIGPIPE


//----------------------------------------------------------------------
class cMessageProcessor {
public:
    virtual int access_level(char *ip_address_string);
    virtual int process_message_buffer(int filedes, char *buffer, int nbytes);
    virtual void doBackground(void);
};


#define MAXMSG	512
#define MAX_CONNECTIONS 40
/* We may need more than one read() for a line. Each connection thus
 * needs its own receive buffer where it can accumulate data until it
 * can be processed.
 */
typedef struct {
    int fd; /* socket */
    char buf[MAXMSG]; /* receive buffer */
    int buf_len; /* number of bytes in buffer */
} connection_t;

int cMessageProcessor::access_level(char *ip_address_string)
{
    return 2;  // 2 == r/w, 1=r, <1, no access, close connection
}  // access_level

int
cMessageProcessor::process_message_buffer(int filedes,
					  char *buffer,
					  int nbytes)
{
#define NWORDS 10
    char *words[NWORDS];
    const char delims[] = " \t\n\r";  // xxx: may add ",="???
    int number_of_words = 0;
    char *tok;
    char *running;
    int retCode;

    if (nbytes < 1) {
	return 0;  // null request, processed okay
    }

    buffer[nbytes-1] = '\0';  // ensure that 'buffer' ends as a string
    // Make a copy since 'strsep()' modifies its buffer (puts in '\0' chars).
    assert( (running = strdup(buffer)) != NULL );
    while ( (tok = strsep(&running, delims)) ) {
	if (number_of_words < NWORDS) {
	    words[number_of_words] = tok;
	    number_of_words++;
	}
    }  // while tokens available

    if (number_of_words < 1) {
	retCode = 0;  // null request, processed okay
	goto lExit;
    }

    if (strcmp(words[0], "quit")==0) {
	retCode = -1;  // end-of-file, close connection
	goto lExit;
    }
    if (strcmp(words[0], "end")==0) {
	retCode = -1;  // end-of-file, close connection
	goto lExit;
    }
    if (strcmp(words[0], "exit")==0) {
	retCode = -1;  // end-of-file, close connection
	goto lExit;
    }
    if (strcmp(words[0], "GET")==0) {
	// Default reply to an HTTP message.
	const char msg[] =
	    // xxx: actually this is in incoming 'GET' message, not in reply:
	    // "Content-type: text/html\n\n"
	    "<HEAD><TITLE>404 Not Found</TITLE></HEAD>\n"
	    "<BODY><H1>404 Not Found</H1>"
	    "The requested URL was not found on this server.<P>\n"
	    "</BODY>";
	assert ( write (filedes, msg, sizeof(msg)-1) == (sizeof(msg)-1) ); 
	// don't write trailing '\0'
	retCode = -1;  // close connection
	goto lExit;
    }

    // "Unknown message":
    // Sometimes the other end has gone away before we get the chance
    // to write back this message.
    if ( write (filedes, "UN,", 3) != 3 ) {
	perror (MY_NAME "write");
	fprintf (stderr, MY_NAME "writing UN error reply message to fd %d\n", filedes);
	retCode = -1;  // close (broken) connection
	goto lExit;
    }
    if ( write (filedes, buffer, nbytes) != nbytes ) {
	perror (MY_NAME "write");
	fprintf (stderr, MY_NAME "writing UNknown '%*s' to fd %d\n", nbytes, buffer, filedes);
	retCode = -1;  // close (broken) connection
	goto lExit;
    }
    if ( write (filedes, "\n", 1) != 1 ) {
	perror (MY_NAME "write");
	fprintf (stderr, MY_NAME "writing LF to fd %d\n", filedes);
	retCode = -1;  // close (broken) connection
	goto lExit;
    }
    retCode = 0;

  lExit:
    if (running) {
	free(running);
    }
    return retCode;
}  // process_message_buffer

void cMessageProcessor::doBackground(void) {
    // Nothing yet.
}  // doBackground


//----------------------------------------------------------------------
class cSocketServer {
    int allocate_connection(int fd);
    connection_t *lookup_connection(int fd);
    void free_connection(connection_t *c);
    void init_connection_table();
    int do_read(connection_t *c, class cMessageProcessor& mp);
    connection_t connections[MAX_CONNECTIONS];
public:
    void serve_socket(int port_number, class cMessageProcessor& mp);
};

int
cSocketServer::allocate_connection(int fd)
{
    int i;
    for (i = 0; i < MAX_CONNECTIONS; i++) {
	if (this->connections[i].fd == -1) {
	    this->connections[i].fd = fd;
	    return i;
	}
    }
    return -1;
}

connection_t *
cSocketServer::lookup_connection(int fd)
{
    int i;
    for (i = 0; i < MAX_CONNECTIONS; i++) {
	if (this->connections[i].fd == fd) {
	  return &this->connections[i];
	}
    }
    return NULL;
}

void
cSocketServer::free_connection(connection_t *c) {
    c->fd = -1;
    memset(c->buf, 0, MAXMSG);
    c->buf_len = 0;
}

void
cSocketServer::init_connection_table(void)
{
  int i;
  for (i = 0; i < MAX_CONNECTIONS; i++) {
      free_connection(&this->connections[i]);
  }
}

/* Called when select indicates that we can read from this
 * connection. Read data to receive buffer and call
 * process_message_buffer if we have a complete line. */
int
cSocketServer::do_read(connection_t *c, class cMessageProcessor& mp)
{
    int nbytes, ret;
    char *newline;
    
    assert(memchr(c->buf, '\n', c->buf_len) == NULL);
    assert(c->buf_len >= 0);
    assert(c->buf_len < MAXMSG);

    nbytes = read(c->fd,
		  c->buf + c->buf_len,
		  MAXMSG - c->buf_len);
    if (nbytes < 0) {
	perror (MY_NAME "read");
	fprintf (stderr, MY_NAME "returning EOF from fd %d\n", c->fd);
	return -1;
    } else if (nbytes == 0) {
	// End-of-file.
	return -1;
    }
    c->buf_len += nbytes;

    newline = (char*)memchr(c->buf, '\n', c->buf_len);
     if (c->buf_len == MAXMSG && newline == NULL) {
	fprintf(stderr, MY_NAME "line too long\n");
	c->buf_len = 0;
	return -1;
    }

    while (newline) {
	int line_len = newline - c->buf + 1 /*newline included*/;
	ret = mp.process_message_buffer(c->fd,
					c->buf,
					line_len);
	memmove(c->buf,
		newline+1,
		MAXMSG - line_len);

	c->buf_len -= line_len;
	if (ret != 0) {
	    return ret;
	}
	newline = (char*)memchr(c->buf, '\n', c->buf_len);
    }
    return 0;
}

static int
make_socket (unsigned short int port)
{
    int sock, one = 1;
    struct sockaddr_in name;

    // Create the socket.
    sock = socket (PF_INET, SOCK_STREAM, 0);
    if (sock < 0) {
	perror (MY_NAME "socket");
	exit (EXIT_FAILURE);
    }

    if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *)&one, sizeof(one)) < 0) {
	perror("setsockopt");
	exit (EXIT_FAILURE);
    }

    // Give the socket a name.
    name.sin_family = AF_INET;
    name.sin_port = htons (port);
    name.sin_addr.s_addr = htonl (INADDR_ANY);
    if (bind (sock, (struct sockaddr *) &name, sizeof (name)) < 0) {
	perror (MY_NAME "bind");
	exit (EXIT_FAILURE);
    }

    return sock;
}  // make_socket

void
cSocketServer::serve_socket (int port_number, class cMessageProcessor& mp)
{
    void (*old_sigpipe_handler)(int) = NULL;
    int sock;
    fd_set active_fd_set, read_fd_set;
    int sfd, ret;

    this->init_connection_table();
    
    // From libc Info file:
    // "Every library function that returns this error code also
    // generates a `SIGPIPE' signal; this signal terminates the program
    // if not handled or blocked."
    // After ignoring we'll only get EPIPE from library calls.
    assert( (old_sigpipe_handler = signal(SIGPIPE, SIG_IGN)) != SIG_ERR );

    // Create the socket and set it up to accept connections.
    sock = make_socket (port_number);
    if (listen (sock, 1) < 0) {
	perror (MY_NAME "listen");
	exit (EXIT_FAILURE);
    }

    // Initialize the set of active sockets.
    FD_ZERO (&active_fd_set);
    FD_SET (sock, &active_fd_set);

    while (1) {
	int numOfDescr;
	struct timeval timeout;

	// Block until input arrives on one or more active sockets.
	read_fd_set = active_fd_set;
	timeout.tv_sec = 0;
	timeout.tv_usec = 10000;  // 10 msec
	if ( (numOfDescr = select (FD_SETSIZE, &read_fd_set, NULL, NULL, &timeout)) < 0 ) {
	    perror (MY_NAME "select");
	}

	// Service all the sockets with input pending.
	if (numOfDescr > 0) {
	    for (sfd = 0; sfd < (int)FD_SETSIZE; ++sfd) {

		// Skip file descriptors with no data pending.
		if (!FD_ISSET (sfd, &read_fd_set)) {
		    continue;
		}

		if (sfd == sock) {
		    // Connection request on original socket.

		    struct sockaddr_in clientname;
		    socklen_t size;
		    int new_sock;

		    size = sizeof (clientname);
		    new_sock = accept (sock,
				       (struct sockaddr *) &clientname,
				       &size);
		    if (new_sock < 0) {
			perror (MY_NAME "accept");
			continue;
		    }
#if DEBUG_CONNECTIONS
		    fprintf (stderr, MY_NAME "connect from host %s, port %hd, (fd %d), ",
			     inet_ntoa (clientname.sin_addr),
			     ntohs (clientname.sin_port),
			     new_sock);
#endif
		    // Connecting from this IP address allowed?
		    if (mp.access_level(inet_ntoa (clientname.sin_addr)) < 1) {
#if DEBUG_CONNECTIONS
			fprintf (stderr, "rejected.\n");
#endif
			(void)close (new_sock);
		    } else {
#if DEBUG_CONNECTIONS
			fprintf (stderr, "accepted.\n");
#endif

			ret = this->allocate_connection(new_sock);
			if (ret < 0) {
			    fprintf(stderr, "cannot allocate connection\n");
			    (void)close (new_sock);
			    continue;
			}
			FD_SET (new_sock, &active_fd_set);
		    }
		} else {
		    // Data arriving on an already-connected socket.
		    connection_t *conn;
		    conn = this->lookup_connection(sfd);
		    assert(conn);

		    if (this->do_read(conn, mp) < 0) {

			// xxx: Some way to exit from this loop and terminate the server?
			//      (close all connections first)
			// xxx: Detecting 'reload' => close all connections? (necessary?)

			(void)close (sfd);
			FD_CLR (sfd, &active_fd_set);
			free_connection(conn);
#if DEBUG_CONNECTIONS
			fprintf (stderr, MY_NAME "disconnected fd %d.\n",
				 sfd);
#endif
		    }
		}  // else data arriving from an existing connection
	    }  // for each file descriptor
	}  // if any file descriptor is ready

	// Now (regardless if we serviced a network request or
	// if only a simple timeout occurred) we process one pending
	// background/logging request.
	mp.doBackground();

    }  // while forever

}  // serve_socket
