// cSerialLine.hh -- a class for handling serial lines.
// $Id: cSerialLine.hh,v 1.6 2002/12/18 07:30:38 amn Exp $
// Copyright (C) 1995 Ari Mujunen. (amn@nfra.nl, Ari.Mujunen@hut.fi)

// This is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License.
// See the file 'COPYING' for details.

/*
 * $Log: cSerialLine.hh,v $
 * Revision 1.6  2002/12/18 07:30:38  amn
 * Upped revision.
 *
 * Revision 1.2  2002/12/18 07:30:37  amn
 * intermediate
 *
 * Revision 1.1  2002/12/18 07:23:46  amn
 * Recovered from /home disk crash by re-creating 'common' module in CVS.
 *
 * Revision 1.5  1998/11/18 12:48:36  jen
 * Adapted for glibc6 (hamm).
 *
 * Revision 1.4  1998/05/29 05:51:08  amn
 * Comment leader of log lines changed.
 *
 * Revision 1.3  1997/03/11 13:46:22  amn
 * Added 'readUntilTimeout()' brute-force read for unknown number of bytes.
 *
 * Revision 1.2  1995/06/16 15:14:24  amn
 * '.h' -> '.hh' for C++ headers.
 *
 * Revision 1.1  1995/02/11  19:54:10  amn
 * Initial revision
 */


#ifndef _cSerialLine_hh
#ifdef __GNUG__
#pragma interface
#endif
#define _cSerialLine_hh 1

#include "registration.h"
RCSID(cSerialLine_hh, "$Id: cSerialLine.hh,v 1.6 2002/12/18 07:30:38 amn Exp $");


// 'cSerialLine' inherits from 'cIOLine' the capabilities to operate on
// devices that are accessed via a Unix file descriptor,
// so we need its header.
#include "cIOLine.hh"

// Whether we want even, odd, or no parity for both input and output.
// Also we can specify that the parity setting should not be changed.
typedef enum {eParityEven,
	      eParityOdd,
	      eParityNone,
	      eParityNoChange
	      } tParity;

// This class tries to abstract the "raw-mode" (no conversions to bytes
// sent/received) bidirectional (read and write) serial interface.
class cSerialLine : public cIOLine
// xxx: if public -> protected, private => open/close not avail as public
//      if omitted => as if private was written here
{
public:
  // The 'open' and 'close' methods of 'cIOLine' are available.

  // The method to change the serial line settings.
  // All parameters can be set to -1 which means "don't change this".
  // You can ask a 'baudRate' in the range 50..115200 baud and
  // the nearest rate supported by hardware will be selected.
  // Number of data bits can be 5..8 and stop bits 1..2.
  // You can ask all 'read' requests
  // to wait for at least 'minBytes' to arrive.
  // (Generally you will want 'read' to return eventually even if no
  //  bytes are received, so you should set 'minBytes' to 0.)
  // If you ask for 'maxTimeout' value of 2, all 'read' requests will wait
  // for up to 0.2 seconds for "yet another byte" to arrive to fulfill
  // the 'count' requested.
  virtual void serialSettings(int baudRate,
			      int numberOfDataBits,
			      int numberOfStopBits,
			      tParity whichParity,
			      int minBytes,
			      int maxTimeout
			      );

  // This promises to read at most 'count' bytes from the serial line
  // and return the number of bytes actually read.
  // (If returned number is less than 'count', it means we timeouted.)
  virtual int read(void *buffer, int count);
  virtual void readStatisticsToStderr(void);

  // This reads one byte at a time until a timeout occurs.
  // A foolish way to implement reading an unknown number of response bytes.
  // Discards input which it cannot fit into the buffer provided.
  virtual int readUntilTimeout(void *buffer, int count);
  virtual int readLine(void *buffer, int maxCount, int timeouts);

  // This promises to write exactly 'count' bytes to the serial line.
  virtual void write(const void *buffer, int count);

  virtual void defaultSettings(void);
  virtual void flushPendingInput(void);
  virtual void sendBreak(int duration);
  virtual void changeParity(tParity whichParity);
  virtual void changeBaudRate(int baudRate);
};  // class cSerialLine

#endif
