// cSerialLine.cc -- a class for handling serial lines.
// $Id: cSerialLine.cc,v 1.11 2002/12/18 07:30:06 amn Exp $
// Copyright (C) 1995 Ari Mujunen. (amn@nfra.nl, Ari.Mujunen@hut.fi)

// This is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License.
// See the file 'COPYING' for details.

// $Log: cSerialLine.cc,v $
// Revision 1.11  2002/12/18 07:30:06  amn
// Upped revision.
//
// Revision 1.2  2002/12/18 07:30:05  amn
// intermediate
//
// Revision 1.1  2002/12/18 07:23:46  amn
// Recovered from /home disk crash by re-creating 'common' module in CVS.
//
// Revision 1.10  1999/10/07 09:20:11  amn
// Removed Digiboard support temporarily.
//
// Revision 1.9  1998/11/18 12:48:35  jen
// Adapted for glibc6 (hamm).
//
// Revision 1.8  1997/03/12 07:34:16  amn
// First level of ADAM support, 'readUntilTimeout()' for blind reading
// of any response regardless of its length.
//
// Revision 1.7  1995/07/11  17:25:29  amn
// Added six-byte debug output in 'read()'.
//
// Revision 1.6  1995/06/29  14:36:31  amn
// DigiBoard-specific >38400 baud settings.
//
// Revision 1.5  1995/06/25  15:29:28  amn
// Sending break refined.
//
// Revision 1.4  1995/06/18  14:16:22  amn
// 'serialSettings()' now waits for transmitter buffer empty,
// a histogram for this waiting (a ka 'read()' delay histogram.)
//
// Revision 1.3  1995/06/16  15:14:10  amn
// Correction of 'read()' not to thrash outside its buffer; '.hh'.
//
// Revision 1.2  1995/03/30  11:02:38  amn
// Changing test for success from != -1 to == 0: should be safer to
// assume everything else than zero mean failure.
//
// Revision 1.1  1995/02/11  19:43:33  amn
// Initial revision
//

#include "registration.h"
RCSID(cSerialLine_cc, "$Id: cSerialLine.cc,v 1.11 2002/12/18 07:30:06 amn Exp $");


#pragma implementation
#include "cSerialLine.hh"

#include <unistd.h>
#include <termios.h>
#include <assert.h>
#include <errno.h>
#include <sys/ioctl.h>
#ifdef TIOCSSERIAL
#include <sys/errno.h>
#include <linux/serial.h>  // in Linux >= 1.1.59
#define SUPPORT_DIGI 0
#if SUPPORT_DIGI
#include "/usr/src/linux/drivers/char/digi.h"  // yechh!
#endif
#endif  // TIOCSSERIAL

#include <stdio.h>  // for fprintf() in statistics


// Forget those characters that might have arrived while we were busy
// doing other things.
void
cSerialLine::flushPendingInput(void)
{
  assert(tcflush(fileDesc, TCIFLUSH) == 0);
}  // flushPendingInput

// Keep break condition on serial line for
// duration * (0.25..0.5 seconds).
// (This is what 'tcsendbreak()' promises.)
void
cSerialLine::sendBreak(int duration)
{
  // Ok, first we try a vanilla Posix-compliant call.
  if (tcsendbreak(fileDesc, duration) == 0) {
    return;
  }
  if (errno != EINVAL) {
    perror("tcsendbreak(fileDesc, duration)");
    assert(errno == EINVAL);
    return;
  }

  // Oops, 'tcsendbreak()' didn't support sending break
  // of a specified length.  Try a direct ioctl():
#ifdef TCSBRKP
  if (ioctl(fileDesc, TCSBRKP, &duration) == 0) {
    return;
  }
#endif
  // Oh no, this failed, too; try at least "plain" break:
  assert(tcsendbreak(fileDesc, 0) == 0);
}  // sendBreak

void
cSerialLine::defaultSettings(void)
{
  // We want to make sure that the newly-opened file is actually
  // capable of accepting termios controls.
  assert(isatty(fileDesc));

  serialSettings(9600,
		 8,
		 1,
		 eParityNone,
		 0,  // minBytes
		 0   // maxTimeout
		 );
}  // defaultSettings

void
cSerialLine::changeParity(tParity whichParity)
{
  serialSettings(-1,
		 -1,
		 -1,
		 whichParity,
		 -1,
		 -1
		 );
}  // changeParity

void
cSerialLine::changeBaudRate(int baudRate)
{
  serialSettings(baudRate,
		 -1,
		 -1,
		 eParityNoChange,
		 -1,
		 -1
		 );
}  // changeBaudRate


// "Transmitter empty" wait loop histogram.
#define SLOTS_IN_TEMT_HISTOGRAM 32
static int waitOfThisManyTimes[SLOTS_IN_TEMT_HISTOGRAM] = {0, };


void
cSerialLine::serialSettings(int baudRate,
			    int numberOfDataBits,
			    int numberOfStopBits,
			    tParity whichParity,
			    int minBytes,
			    int maxTimeout
			    )
{
  struct termios settings;

  assert((baudRate < 0) || ((baudRate >= 50) && (baudRate <= 115200)));
  assert((numberOfDataBits < 0)
            || ((numberOfDataBits >= 5) && (numberOfDataBits <= 8)));
  assert((numberOfStopBits < 0)
            || ((numberOfStopBits >= 1) && (numberOfStopBits <= 2)));

  // The recommended way to alter termios settings is
  // 1) get previous settings
  // 2) alter them; touch only those bits that need changing
  // 3) set the new settings.
  // So, we start by getting the current settings:
  assert(tcgetattr(fileDesc, &settings) == 0);

  // Input mode flags.
  settings.c_iflag
    &= ~(
     INPCK   // no checking for parity errors on input
    |IGNPAR  // don't ignore bytes with framing or parity errors
             // (needs INPCK)
    |PARMRK  // don't generate the sequence '377,0,err_byte' for bytes
             // with parity errors (needs INPCK)
    |ISTRIP  // don't strip 8th bit
    |BRKINT  // don't generate SIGINT from incoming BREAK (needs ~IGNBRK)
    |IGNCR   // don't ignore CRs ('\r')
    |ICRNL   // dont' translate CRs -> LFs
    |INLCR   // dont' translate LFs -> CRs
    |IXOFF   // don't enable start/stop on input
    |IXON    // don't enable start/stop on output
    |IMAXBEL // don't send BELs to indicate filling of input buffer
    |IUCLC   // don't do upper/lowercase conversions
  );
  settings.c_iflag
    |= (
     IGNBRK   // ignore incoming BREAK conditions
    |IXANY    // start with any character (not just the start character)
  );

  // Output mode flags.
  settings.c_oflag
    &= ~(
     OPOST   // no output "post-processing", characters transmitted as-is
    |ONLCR   // no '\n' -> '\r\n' conversion (needs OPOST)
#ifdef OXTABS
    |OXTABS  // no '\t' -> spaces conversion (needs OPOST)
#else
    |XTABS   // Linux/BSD? convention?
#endif
#ifdef ONOEOT
    |ONOEOT  // don't discard ctrl-D characters in output (needs OPOST)
#endif
    |OLCUC   // don't do upper/lowercase conversions
  );
    // Linux/BSD?: OCRNL, ONOCR, ONLRET, OFILL, OFDEL, NLDLY, CRDLY,
    //             TABDLY, (XTABS is a submask of TABDLY),
    //             BSDLY, VTDLY, FFDLY

  // Control mode flags.
  {
    int disableThese = 0;
    int enableThese = 0;

    // Select number of data bits per each byte.
    switch (numberOfDataBits) {
      case 5: {
        enableThese |= CS5;
	disableThese |= (CSIZE & ~CS5);
        break;
      }
      case 6: {
        enableThese |= CS6;
	disableThese |= (CSIZE & ~CS6);
        break;
      }
      case 7: {
        enableThese |= CS7;
	disableThese |= (CSIZE & ~CS7);
        break;
      }
      case 8: {
        enableThese |= CS8;
	disableThese |= (CSIZE & ~CS8);
        break;
      }
      default: {
        // nothing, leave bits as they are
        break;
      }
	    }  // switch #of data bits

    // Select number of stop bits at the end of each byte.
    switch (numberOfStopBits) {
      case 1: {
        disableThese |= CSTOPB;
        break;
      }
      case 2: {
        enableThese |= CSTOPB;
        break;
      }
      default: {
        // nothing, leave bits as they are
        break;
      }
	    }  // switch #of stop bits

    // Select the polarity of parity bits at the end of each byte.
    switch (whichParity) {
      case eParityEven: {
        disableThese |= PARODD;
        enableThese |= PARENB;
        break;
      }
      case eParityOdd: {
        enableThese |= PARODD;
        enableThese |= PARENB;
        break;
      }
      case eParityNone: {
        disableThese |= PARENB;
        break;
      }
      default: {
        // nothing, leave bits as they are
        break;
      }
		      }  // switch parity polarity

    settings.c_cflag
      &= ~(
       HUPCL    // don't generate modem disconnect when closed
      |CRTSCTS  // don't enable hardware CTS-output/RTS-input flow control
      |disableThese
    );
    settings.c_cflag
      |= (
       CLOCAL  // "locally connected", ignore modem status lines
      |CREAD   // input can be read (input is not discarded)
      |enableThese
    );
  }  // block for setting control flags

  // Local mode flags.
  settings.c_lflag
    &= ~(
     ICANON  // input is processed in non-canonical mode (not by line)
    |ECHO    // don't echo input characters back
    |ISIG    // don't recognize INTR, QUIT, and SUSP characters
    |IEXTEN  // don't recognize implementation-defined special characters
    |TOSTOP  // don't generate SIGTTOU job control signals
    |ECHOCTL // (BSD) don't echo control characters as '^A'
#ifdef XCASE
    |XCASE   // Linux/BSD?  Translate case of letters?
#endif
  );
  settings.c_lflag
    |= (
    0  // nothing yet
  );
  // ECHOE irrelevant with ~ICANON
  // ECHOK irrelevant with ~ICANON
  // ECHONL irrelevant with ~ICANON
  // NOFLSH irrelevant because of ~ISIG
  // ECHOKE (BSD) echoing of KILL character
  // ECHOPRT (BSD) echoing of ERASE character on hardcopy terminal
  // FLUSHO (BSD) DISCARD character bit
  // PENDIN (BSD) REPRINT character bit
  // (BSD n/a in Linux: ALTWERASE, NOKERNINFO)

  // Select both input and output baud rates.
  if (baudRate < 0) {
    // No change in baud rate bits.
  } else {
    int baud;

    if (baudRate > 28800) {
      baud = B38400;
    } else if (baudRate > 14400) {
      baud = B19200;
    } else if (baudRate > 7200) {
      baud = B9600;
    } else if (baudRate > 3600) {
      baud = B4800;
    } else if (baudRate > 2100) {
      baud = B2400;
    } else if (baudRate > 1500) {
      baud = B1800;
    } else if (baudRate > 900) {
      baud = B1200;
    } else if (baudRate > 450) {
      baud = B600;
    } else if (baudRate > 250) {
      baud = B300;
    } else if (baudRate > 175) {
      baud = B200;
    } else if (baudRate > 142) {
      baud = B150;
    } else if (baudRate > 122) {
      baud = B134;
    } else if (baudRate > 92) {
      baud = B110;
    } else if (baudRate > 62) {
      baud = B75;
    } else if (baudRate > 25) {
      baud = B50;
    } else {
      baud = B0;  // hang up; set modem lines to disconnect
                  // if they are in use
    }
    // Set first input baud rate to follow output baud rate with 'B0'.
    assert(cfsetispeed(&settings, B0) == 0);
    assert(cfsetospeed(&settings, baud) == 0);
    // (Apparently at least in Linux the latest i/o baud setting
    //  applies to _both_ input and output.)
  }  // else change baud rate bits

  // Line discipline is the normal TTY, no PLIP etc. special.
  settings.c_line = N_TTY;

  // Control characters.  ICANON mode only uses these two.
  if (minBytes >= 0) {
    // Has to receive 'minBytes' to return from 'read()'.
    settings.c_cc[VMIN] = minBytes;
  }
  if (maxTimeout >= 0) {
    // Maximum wait in 'read()', 1 = 0.1sec.
    settings.c_cc[VTIME] = maxTimeout;
  }

  // Now we seem to have a slight problem with parity changes.
  // Linux kernels up to and including 1.2.9 wait only
  // for the kernel character queue to drain.  They do not check
  // the UART transmitters.  And 16x50 UARTs perform the
  // parity change immediately, even while still transmitting.

  // Fortunately, in 1.1.63 (and up) there is a way around this:
  // a new 'ioctl()' is available, so you can wait for
  // UART_LSR_TEMT bit to go up and signify "transmitter empty".

  {
    int waitLoopsForTEMT = 0;
    unsigned int result;
    
    while (1) {
      if (ioctl(fileDesc, TIOCSERGETLSR, &result) != 0) {
	if (errno == EINVAL) {
          // Ok, this serial port doesn't support 'TIOCSERGETLSR'.
	  // It might be one of those which don't need this (Digiboard)
	  // or one of those which don't yet support it (Cyclades)...
	  // Nevertheless, try to continue with changing settings.
	  break;
	} else {
	  perror("ioctl(fileDesc, TIOCSERGETLSR, &result)");
	  assert(errno == EINVAL);
	}
      }
      if (result == TIOCSER_TEMT) {
        break;
      }
      // would be tempting to sleep a little---but that becomes easily
      // 10ms at least...
      waitLoopsForTEMT++;
    }
    waitOfThisManyTimes[(waitLoopsForTEMT > SLOTS_IN_TEMT_HISTOGRAM-1) ? SLOTS_IN_TEMT_HISTOGRAM-1 : waitLoopsForTEMT]++;
  }

  // Change serial settings.
  assert(tcsetattr(fileDesc, TCSADRAIN, &settings) == 0);

#ifdef TIOCSSERIAL
  // We attempt to adjust 38400 baud into higher speeds.
  // Linux can give 57600 or 115200 when std Unix 'termios' call
  // requests for 38400.
  // Now we need to adjust the special bits whenever "slower"
  // baud rates are requested, too: namely DigiBoard wants
  // to reuse 50/75/110 baud and to get _these_ slow rates,
  // we must cancel the fastbaud bit...
  if (baudRate > 0)
  {
    struct serial_struct allSerialSettings;
    int oldBits;

    if (ioctl(fileDesc, TIOCGSERIAL, &allSerialSettings) == 0) {
      // We can do the change in Linux 'serial.c' way.
      oldBits = allSerialSettings.flags & ASYNC_SPD_MASK;

      // Zero the SPD bits first.  (== "normal" 38400 baud)
      allSerialSettings.flags &= ~ASYNC_SPD_MASK;

      if (baudRate > 86400) {
        // 115200 baud.
        allSerialSettings.flags |= ASYNC_SPD_VHI;
      } else if (baudRate > 48000) {
        // 57600 baud.
        allSerialSettings.flags |= ASYNC_SPD_HI;
      } else {
        // Leave at genuine 38400 baud (or lower.)
      }

      // Change (all the) serial settings only if they really need changing.
      if ((allSerialSettings.flags & ASYNC_SPD_MASK) != oldBits) {
        assert(ioctl(fileDesc, TIOCSSERIAL, &allSerialSettings) == 0);
      }
    } else {
#if SUPPORT_DIGI
      if (errno == EINVAL) {
        // 'TIOCGSERIAL' didn't work, perhaps a Digiboard special will?
        digi_t digiSettings;
        digi_t oldDigiSettings;
	struct termios oldSettings;

        // We start by re-getting the current termios + getting Digi settings:
        assert(tcgetattr(fileDesc, &settings) == 0);
	oldSettings = settings;
	assert(ioctl(fileDesc, DIGI_GETA, &digiSettings) == 0) ;
	oldDigiSettings = digiSettings;

        if (baudRate > 86400) {
          // 115200 baud.
          digiSettings.digi_flags |= DIGI_FAST;
          assert(cfsetispeed(&settings, B0) == 0);
          assert(cfsetospeed(&settings, B110) == 0);
        } else if (baudRate > 67200) {
          // 76800 baud.
          digiSettings.digi_flags |= DIGI_FAST;
          assert(cfsetispeed(&settings, B0) == 0);
          assert(cfsetospeed(&settings, B75) == 0);
        } else if (baudRate > 48000) {
          // 57600 baud.
          digiSettings.digi_flags |= DIGI_FAST;
          assert(cfsetispeed(&settings, B0) == 0);
          assert(cfsetospeed(&settings, B50) == 0);
        } else {
          // Leave at genuine 38400 baud (or lower.)
          digiSettings.digi_flags &= ~DIGI_FAST;
	  // (The baud rate has been already set in standard settings
	  //  above.)
        }

        // Change Digi settings + termios settings again,
	// if either Digi or std. baud rate changed.
	if (digiSettings.digi_flags != oldDigiSettings.digi_flags) {
	  assert(ioctl(fileDesc, DIGI_SETAW, &digiSettings) == 0);
	}
	if (settings.c_cflag != oldSettings.c_cflag) {
	  assert(tcsetattr(fileDesc, TCSADRAIN, &settings) == 0);
	}
      } else {
	// Getting Linux-specific serial settings failed
	// in other way than EINVAL...
#endif
	perror("ioctl(fileDesc, TIOCGSERIAL, &allSerialSettings)");
#if SUPPORT_DIGI
	assert(errno == EINVAL);
      }
#endif
    }
  }  // if > 0 baud
#endif // TIOCSSERIAL
}  // serialSettings


#define SLOTS_IN_READ_HISTOGRAM 32
static int readsOfThisDelay[SLOTS_IN_READ_HISTOGRAM] = {0, };
#include <sys/times.h>


// Tries to read all 'count' bytes.
// If returns less than 'count' bytes, a timeout must have occurred.
int
cSerialLine::read(void *buffer, int count)
{
  ssize_t stillUnread;
  ssize_t readThisTime;
  clock_t startTimeOfThisRead;
  int delayOfThisRead;

  stillUnread = count;
  startTimeOfThisRead = times(NULL);
  do {
    readThisTime = ::read(fileDesc, buffer, stillUnread);

// Note: Linux kernel versions <= 1.0.8 can sometimes return zero bytes.
#if DEBUG
    printf(
      "cSerialLine: read(fd=%d, buff=%08X (%02X %02X %02X %02X %02X %02X), count=%d) => readThisTime=%d\n",
      fileDesc,
      (int)buffer,
      (int)((unsigned char *)buffer)[0],
      (int)((unsigned char *)buffer)[1],
      (int)((unsigned char *)buffer)[2],
      (int)((unsigned char *)buffer)[3],
      (int)((unsigned char *)buffer)[4],
      (int)((unsigned char *)buffer)[5],
      (int)stillUnread,
      (int)readThisTime
    );
#endif

    // 'read()' must not fail.
    assert(readThisTime != -1);

    // Decrement amount of bytes to try, advance buffer pointer as bytes.
    stillUnread -= readThisTime;
    buffer = (unsigned char *)buffer + readThisTime;

  } while ((stillUnread > 0) && (readThisTime > 0));

  delayOfThisRead = times(NULL) - startTimeOfThisRead;
  readsOfThisDelay[(delayOfThisRead > SLOTS_IN_READ_HISTOGRAM-1) ? SLOTS_IN_READ_HISTOGRAM-1 : delayOfThisRead]++;

  return ((int)(count - stillUnread));
}  // read

int
cSerialLine::readUntilTimeout(void *buffer, int count)
{
  char *p = (char *)buffer;
  int c = 0;

  while ( read(p, 1) == 1 ) {
    p++;
    c++;
    if (c >= count) {
      // Cannot put more into buffer provided.
      flushPendingInput();
      break;
    }
  }  // while
  return (c);
}  // readUntilTimeout

int
cSerialLine::readLine(void *buffer, int maxCount, int timeouts)
{
  char *p = (char *)buffer;
  int totalBytes = 0;
  int rcvBytes;

  // Retrieve one line (ending in '\n').
  do {
    rcvBytes = this->read(p, maxCount);
    p[rcvBytes] = '\0';
    if (rcvBytes <= 0) {  // no response at all
      timeouts--;
      if (timeouts <= 0) {
	((char *)buffer)[0] = '\0';  // no response, no line
	return 0;
      }
      continue;
    }
    totalBytes += rcvBytes;
    if (p[rcvBytes-1] == '\n') {
      return totalBytes;  // normal end of line
    }
    p += rcvBytes;
    maxCount -= rcvBytes;
  } while (1);
}  // readLine

// Tries to write all 'count' bytes.
void
cSerialLine::write(const void *buffer, int count)
{
  ssize_t bytesWritten;

  assert((bytesWritten = ::write(fileDesc, buffer, (ssize_t)count)) != -1);
#if DEBUG
    printf(
      "cSerialLine: write(fd=%d, buff=%08X (%02X %02X %02X %02X %02X %02X), count=%d) => bytesWritten=%d\n",
      fileDesc,
      (int)buffer,
      (int)((unsigned char *)buffer)[0],
      (int)((unsigned char *)buffer)[1],
      (int)((unsigned char *)buffer)[2],
      (int)((unsigned char *)buffer)[3],
      (int)((unsigned char *)buffer)[4],
      (int)((unsigned char *)buffer)[5],
      (int)count,
      (int)bytesWritten
    );
#endif
  assert(bytesWritten == count);
  return;
}  // write


// Writes out statistics of read call delays.
void
cSerialLine::readStatisticsToStderr(void)
{
  int i;
  int totals;

  // TEMT histogram.
  totals = 0;
  for (i=0; i<SLOTS_IN_TEMT_HISTOGRAM; i++) {
    fprintf(stderr, "%02d: %d\n", i, waitOfThisManyTimes[i]);
    totals += waitOfThisManyTimes[i];
  }
  fprintf(stderr, "To: %d\n\n", totals);

  // 'read()' delay histogram.
  totals = 0;
  for (i=0; i<SLOTS_IN_READ_HISTOGRAM; i++) {
    fprintf(stderr, "%02d0ms: %d\n", i, readsOfThisDelay[i]);
    totals += readsOfThisDelay[i];
  }
  fprintf(stderr, "Total: %d\n", totals);
}  // readStatisticsToStderr
