/*************************************************
 *                                               *
 * VDIF Extract for thread payload extraction    *
 *                                               *
 *************************************************/

#include "libvdif.h"
#define _GNU_SOURCE 1
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#define _XOPEN_SOURCE 500
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#define VDIFDMP_VER_STR "VDIF Extract Version 1.0"
#define VDIFDMP_CPY_STR "(C) 2009 Jan Wagner, GNU GPL v3"

typedef unsigned long long ull_t;

void print_usage() 
{
   printf(" Usage: vdif_extract threadID infile outfile\n\n"
          "    threadID : the thread (0..X) to extract\n"
          "    infile   : the input VDIF file\n"
          "    outfile  : the output data file\n"
          " Missing frames written as random or filesystem data.\n");
}

int main(int argc, char* argv[]) 
{
   char frame[1024];

   char* filename_in;
   char* filename_out;
   int fd_in, fd_out;

   off64_t framecount = 0, readpos = 0, second = 0;
   off64_t max_framenr = 0;
   off64_t first_framenr = (unsigned)-1;
   off64_t min_second = (unsigned)-1;
   vdifheader_parsed_t header;
   int threadID;

   /* command line arguments */
   printf("\n %s\n %s\n\n", VDIFDMP_VER_STR, VDIFDMP_CPY_STR);
   if (argc < 4) {
      print_usage(); 
      return 0;
   }
   threadID = atoi(argv[1]);
   filename_in = argv[2];
   filename_out = argv[3];
   
   /* open the files */
   fd_in = open(filename_in, O_RDONLY);
   if (fd_in < 0) {
      printf("Could not open input file: '%s'\n", strerror(errno)); 
      return -1;
   }
   fd_out = creat(filename_out, S_IRUSR|S_IWUSR|S_IRGRP);
   if (fd_out < 0) {
      printf("Could not open/create output file: '%s'\n", strerror(errno)); 
      return -1;
   }

   /* 1st pass: scan the headers plus a little bit of payload data */
   printf("Pass 1: analysis...\n");
   while (1) {
      ssize_t n;
      off64_t old_readpos;

      // read header + some data
      n = pread(fd_in, frame, sizeof(frame), readpos); 
      if (n < 16) { break; }
      vdif_parse_header(frame, &header);
      old_readpos = readpos;
      readpos += 8*header.framelength_w64s;
      if (header.threadID != threadID) {
          continue;
	  }

      // analysis
      if (header.seconds > second) {
         second = header.seconds;
         printf("Second %Lu at byte 0x%08llX, intra-sec frame #%Lu, length %Lu byte\n", 
               (ull_t)second, (ull_t)old_readpos, (ull_t)header.frame,
               (ull_t)(8ULL*header.framelength_w64s));
      }
      if (header.frame > max_framenr) {
         max_framenr = header.frame;
      }
      if (header.seconds < min_second) {
         min_second = header.seconds;
      }
      if (first_framenr == (unsigned)-1) {
         first_framenr = header.frame;
      }

      framecount++;
   }
   printf("File had %Lu frames of thread ID %u, smallest seconds stamp %Lu, highest frame nr %Lu, first frame nr %Lu.\n", 
         (ull_t)framecount, threadID, (ull_t)min_second, 
         (ull_t)max_framenr, (ull_t)first_framenr);
   if (framecount == 0) {
      close(fd_in);
      close(fd_out);
      return 0;
   }

   /* 2nd pass: extract the payload */
   printf("Pass 2: extraction...\n");
   readpos = 0;
   second = 0;
   first_framenr = 0; // "clear" to 0 if files should start at integer seconds
   while (1) {
      ssize_t n;
      off64_t writepos, old_readpos;

      // read header + some data
      n = pread(fd_in, frame, sizeof(frame), readpos); 
      if (n < 16) { break; }
      vdif_parse_header(frame, &header);
      old_readpos = readpos;
      readpos += 8*header.framelength_w64s;
      if (header.threadID != threadID) {
          continue;
	  }

      // write at correct output position
      writepos = ( (header.seconds - min_second) * max_framenr 
                   + header.frame) * header.payloadlength_bytes
                 - first_framenr * header.payloadlength_bytes;
      pwrite(fd_out, header.payload, header.payloadlength_bytes, writepos);

      if (header.seconds != second) {
         second = header.seconds;
         printf("Second %Lu\n", (ull_t)second);
      }
#if 0
      printf("Read %u from %08LX write to %08LX\n", 
             header.payloadlength_bytes, 
             (ull_t)old_readpos, (ull_t)writepos);
#endif
   }

   close(fd_in);
   close(fd_out);

   return 0;
}
