#ifndef LIBVDIF_H
#define LIBVDIF_H

/***************************************************
 **                                               **
 ** A "library" for processing simple VDIF data.  **
 ** Implements VDIF Release 1.0 6th March 2009    **
 **                                               **
 ** (C) 2009 Jan Wagner, Metsahovi Radio Obs      **
 **                                               **
 ** Licencing: GNU GPL v3                         **
 **                                               **
 ***************************************************/

///////////////////////////////////////////////////
// Definitions and includes
///////////////////////////////////////////////////

#define VDIF_VER_STR "VDIF Library Version 1.0"
#define VDIF_CPY_STR "(C) 2009 Jan Wagner, GNU GPL v3"

#include <sys/types.h>
#include <stdio.h>

///////////////////////////////////////////////////
// VDIF structure
///////////////////////////////////////////////////

#pragma pack(0)
typedef union vdifheader_tt {
    struct raw_t {
        u_int64_t header_qw0;
        u_int64_t header_qw1;
        u_int64_t extdata_qw0;
        u_int64_t extdata_qw1;
    } raw;
    struct bitfield_t {
        // word 0
        unsigned  invalid : 1;
        unsigned  legacy : 1;
        unsigned  epoch_seconds : 30;
        // word 1
        unsigned  _dontcare1 : 2;
        unsigned  epoch : 6;
        unsigned  frame : 24;
        // word #2
        unsigned version : 3;
        unsigned log2channels : 5;
        unsigned framelength_w64s : 24;
        // word #3
        unsigned complex : 1;
        unsigned bitspersample : 5;
        unsigned thread : 10;
        signed stationID0 : 8;
        signed stationID1 : 8;
        // word #4
        unsigned extdataversion : 8;
        unsigned extdata0_i0 : 24;
        // word #5
        unsigned extdata0_i1 : 32;
        // word #6
        unsigned extdata1_i0 : 32;
        // word #7
        unsigned extdata1_i1 : 32;  
    } bitfield;
} vdifheader_t;

typedef struct vdifheader_parsed_tt {
    const char* base;
    const char* payload;
    char invalid;
    char legacy;
    u_int32_t seconds;
    u_int32_t epoch;
    u_int32_t frame;
    int version;
    int log2channels;
    u_int32_t framelength_w64s;
    u_int32_t payloadlength_bytes;
    char complex;
    int bitspersample;
    int threadID;
    char stationID[3]; // "xx\0"
    int extdataversion;
    char extdata[15];
} vdifheader_parsed_t;

///////////////////////////////////////////////////
// Header API Functions
///////////////////////////////////////////////////

void print_vdif(FILE*, vdifheader_t*);
void print_qword_bits(FILE*, const u_int64_t*);

/*
 * Get individual fields from a VDIF Data Frame Header
 */
char vdif_get_invalid(const char*);
char vdif_get_legacy(const char*);
char vdif_get_version(const char*);
char vdif_get_log2channels(const char*);
char vdif_get_bitspersample(const char*);
char vdif_get_complex(const char*);
const char* vdif_get_station(const char*);
const char* vdif_get_payload(const char*);
u_int32_t vdif_get_seconds(const char*);
u_int32_t vdif_get_epoch(const char*);
u_int32_t vdif_get_thread(const char*);
u_int32_t vdif_get_framenumber(const char*);
u_int32_t vdif_get_framelength(const char*);
int vdif_get_extdata_version(const char*);

/*
 * Parse all VDIF Data Frame Header fields into a more
 * readily usable struct that requires no additional
 * bit fiddling.
 */
vdifheader_parsed_t* vdif_parse_header(const char*, vdifheader_parsed_t*);

#endif // LIBVDIF_H
