/*************************************************
 *                                               *
 * UDP VDIF data receiver for iBob streams       *
 *                                               *
 * Listens on the specified port for UDP packets *
 * of user-specified size and writes them to a   *
 * specified output file.                        *
 *                                               *
 * No packet filtering or user header inspection *
 * is done. In a sense, this is a dumb 'tcpdump' *
 * directly to disk.                             * 
 *                                               *
 * Usage: ibobvdifclient port length file        *
 *                                               *
 *   port   : port of incoming UDP data          *
 *   length : how many bytes or seconds to read  *
 *   file   : output file                        *
 *                                               *
 *************************************************/

#include "libvdif.h"
#include "ibobcontrol.h"
#include "ibobvdifclient.h"

#define VDIFCLI_VER_STR "iBob UDP VDIF Client Version 1.0"
#define VDIFCLI_CPY_STR "(C) 2009 Jan Wagner, GNU GPL v3"

int     bind_udp_socket();
void    print_usage();
off64_t parse_length(const char*, int*);
void    SIGINT_handler(int);
double  deltaT_usec(struct timeval, struct timeval);
void    stats_final();
void    stats_current();
void    realtime_init();

static settings_t       cfg;
static ibobcontrol_t    ibob;
static transfer_stats_t stats;
static sig_atomic_t     ctrlC_pressed = 0;

void print_usage() 
{
   printf(" Usage: ibobvdifclient port length file\n\n"
          "    port   : port for incoming UDP/VDIF data\n"
          "    length : how many bytes or seconds to record\n"
          "    file   : the output file\n\n"
          " The length can be specified as 16k, 16M, 16G, 16T\n"
          " or as a time duration 30s, 30m, 30h, or -1 for infinite.\n\n");
}


int main(int argc, char* argv[]) 
{
   struct sigaction sa;
   int status;

   /* clear all */
   memset(&stats, 0, sizeof(stats));
   memset(&cfg, 0, sizeof(cfg));
   memset(&ibob, 0, sizeof(ibob));

   /* command line arguments */
   printf("\n %s\n %s\n\n", VDIFCLI_VER_STR, VDIFCLI_CPY_STR);
   if (argc < 3) {
      print_usage();
      return EXIT_OK;
   }
   cfg.port       = atoi(argv[1]);
   cfg.final_size = parse_length(argv[2], &cfg.size_type);
   cfg.filename   = strdup(argv[3]);
   if (cfg.final_size == 0) { return EXIT_ERROR; }
   if (cfg.size_type == TYPE_BYTES) {
      printf("Recording %Lu bytes\n", (ull_t)cfg.final_size);
   } else if (cfg.size_type == TYPE_SECONDS) {
      printf("Recording %Lu seconds\n", (ull_t)cfg.final_size);
   } else {
      printf("Recording until SIGINT or Ctrl-C\n");
   }

   /* required for multi-gigabit: single-core affinity, RT scheduler */ 
   realtime_init();

   /* allocations */
   cfg.datagram = memalign(128, MAX_UDP_SIZE+1);
   cfg.vdif     = (vdifheader_t*)cfg.datagram;
   if (cfg.datagram == NULL) { return EXIT_ERROR; }
   
   /* open the file */
   cfg.file_fd = open( cfg.filename, 
                       O_RDWR|O_CREAT|O_TRUNC, 
                       S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH );
   if (cfg.file_fd < 0) {
      printf("Could not open/create file: '%s'\n", strerror(errno)); 
      return EXIT_FILE_ERROR;
   }
   if (posix_fadvise(cfg.file_fd, 0, 0, POSIX_FADV_SEQUENTIAL|POSIX_FADV_NOREUSE|POSIX_FADV_DONTNEED) < 0) {
      printf("fadvise() failed: '%s'\n", strerror(errno));
   }
   fsync(cfg.file_fd);

   /* bind to a UDP socket */
   status = bind_udp_socket();
   if (!status) {
      printf("Couldn't get a socket for port %u\n", cfg.port);
      return EXIT_NET_ERROR;
   }

   /* register Control-C / SIGINT handler */
   sigemptyset (&sa.sa_mask);
   sa.sa_flags = 0;
   sa.sa_handler = SIGINT_handler;
   sigaction (SIGINT, &sa, 0);

   /* receive the requested amount of data */
   gettimeofday(&stats.total.start, NULL);
   stats.interval.start = stats.total.start;
   while (!ctrlC_pressed) {
      ssize_t nrd, nwr;
      off64_t second;
      nrd = recv(cfg.socket_fd, cfg.datagram, MAX_UDP_SIZE, 0);
      if (nrd < 0) {
         printf("recv() failed: '%s'\n", strerror(errno));
         // return EXIT_NET_ERROR;
         break;
      }
      stats.total.packets += 1;
      stats.total.bytes   += nrd;
      stats.total.smallest_udp = nrd; // ...
      stats.total.largest_udp  = nrd; // ...
      second = vdif_get_seconds(cfg.datagram);
      if (cfg.base_seconds == 0) {
         cfg.base_seconds = second;
         cfg.previous_vdif_second = second;
         printf("VDIF: first second: %Lu/0x%LX\n", (ull_t)second, (ull_t)second);
         // print_vdif(stdout, cfg.vdif);
      }
#if 1
      nwr = 0;
      do {
         ssize_t w = write(cfg.file_fd, cfg.datagram+nwr, nrd-nwr);
         if (w < 0) {
            printf("write() failed: '%s'\n", strerror(errno));
            break;
         }
         nwr += w;
      } while(nwr<nrd);
#else
      nwr = nrd;
#endif
      if (second != cfg.previous_vdif_second) {
         cfg.previous_vdif_second = second;
         stats_current();
      }
      if (cfg.size_type == TYPE_BYTES) {
         if (stats.total.bytes >= cfg.final_size) break;
      } else if (cfg.size_type == TYPE_SECONDS) {
         if ((second - cfg.base_seconds) >= cfg.final_size) break;
      }
   }
   gettimeofday(&stats.total.stop, NULL);
   if (ctrlC_pressed) {
      printf("Caught control-C\n");
   }

   /* close all */
   free(cfg.datagram);
   close(cfg.socket_fd);
   close(cfg.file_fd);

   /* final stats */
   stats_final();

   return EXIT_OK;
}


// Catch any Control-C -press
void SIGINT_handler(int signum) 
{
   ctrlC_pressed = 1;
}


// Bind to a UDP socket on certain local port
int bind_udp_socket()
{
   struct addrinfo  hints;
   struct addrinfo *info;
   struct addrinfo *info_save;
   char             aibuffer[10];
   int              status = 0;

   memset(&hints, 0, sizeof(hints));
   hints.ai_flags    = AI_PASSIVE;
   hints.ai_family   = AF_INET;
   hints.ai_socktype = SOCK_DGRAM;
   sprintf(aibuffer, "%u", cfg.port);

   getaddrinfo(NULL, aibuffer, &hints, &info);
   info_save = info; 
   do {
      cfg.socket_fd = socket(info->ai_family, info->ai_socktype, info->ai_protocol);
      if (cfg.socket_fd < 0) { continue; }
      if (!bind(cfg.socket_fd, info->ai_addr, info->ai_addrlen)) {
         printf("Receiving UDP on port %u\n", cfg.port);
         status = 1;
         break;
      }
   } while ((info = info->ai_next) != NULL);
   freeaddrinfo(info_save);

   if (status != 0) {
      int sockopt = 16*1024*1024;
      socklen_t socklen;
      setsockopt(cfg.socket_fd, SOL_SOCKET, SO_RCVBUF, &sockopt, sizeof(sockopt));
      getsockopt(cfg.socket_fd, SOL_SOCKET, SO_RCVBUF, (void*)&sockopt, &socklen);
      printf("Socket receive buffer set to %d bytes\n", sockopt);
   }

   return status;
}


// Parse a length string such as "1024G" or "30m"
off64_t parse_length(const char* lengthstr, int* type)
{
   char    *s;
   char    *ptr;
   size_t  slen;
   off64_t value;
   char    unit;

   *type = TYPE_INFINITE;
   if (lengthstr == NULL || type == NULL) { return 0; }
   if ((slen = strlen(lengthstr)) == 0) { return 0; }
   if ((s = strdup(lengthstr)) == NULL) { return 0; }
   if (s[0] == '-') {
      free(s);
      return -1;
   }

   ptr = strpbrk(s, "kMGTsmh\n");
   if (ptr == NULL) {
      printf("Error: length '%s' does not end with any of {k,M,G,s,m,h}\n", s);
      free(s);
      return 0;
   }
   unit  = *ptr;
   *ptr  = '\0';
   value = atoll(s);
   if (unit == 'k') { *type = TYPE_BYTES; value *= 1024UL; }
   if (unit == 'M') { *type = TYPE_BYTES; value *= 1024UL*1024UL; }
   if (unit == 'G') { *type = TYPE_BYTES; value *= 1024UL*1024UL*1024UL; }
   if (unit == 'T') { *type = TYPE_BYTES; value *= 1024UL*1024UL*1024UL; }
   if (unit == 's') { *type = TYPE_SECONDS; }
   if (unit == 'm') { *type = TYPE_SECONDS; value *= 60UL; }
   if (unit == 'h') { *type = TYPE_SECONDS; value *= 3600UL; }
   free(s);
   return value;
}


// Return time difference
double deltaT(struct timeval b, struct timeval e) 
{
   return ( (e.tv_sec - b.tv_sec) + 1e-6*(e.tv_usec - b.tv_usec) );
}


// Final statistics
void stats_final()
{
   double dT = deltaT(stats.total.start, stats.total.stop);
   double Mbps = (8.0 * stats.total.bytes / dT) * 1e-6;
   double goodput_coeff = ((double)stats.total.smallest_udp - 4*8) / (double)stats.total.smallest_udp;
   printf("\n--- Final Statistics ---\n");
   printf("Total bytes     : %.2f MByte\n", ((double)stats.total.bytes)/(1024*1024));
   printf("Total packets   : %Lu\n", (ull_t)stats.total.packets);
   printf("Smallest packet : %u\n", (unsigned int)stats.total.smallest_udp);
   printf("Largest packet  : %u\n", (unsigned int)stats.total.largest_udp);
   printf("Total duration  : %.2f seconds\n", dT);
   printf("Averate rate    : %.2f Mbit/s(dec)\n", Mbps);
   printf("Goodput rate    : %.2f Mbit/s(dec) without VDIF headers\n", Mbps * goodput_coeff);
   return;
}

// Interval statistics
void stats_current()
{
   double dT, dB, Mbps;
   off64_t dS;
   gettimeofday(&stats.interval.stop, NULL);
   dT = deltaT(stats.interval.start, stats.interval.stop);
   dB = stats.total.bytes - stats.interval.bytes;
   Mbps = 8e-6 * dB / dT;
   dS = cfg.previous_vdif_second - cfg.base_seconds;
   printf("-----------------------------------\n");
   printf("Current second  : %Lu\n", (ull_t)dS);
   printf("Current bytes   : %.2f MByte\n", ((double)stats.total.bytes)/(1024*1024));
   printf("Current packets : %Lu\n", (ull_t)stats.total.packets);
   printf("Current rate    : %.2f Mbit/s(dec)\n", Mbps);
   stats.interval.start = stats.interval.stop;
   stats.interval.bytes = stats.total.bytes;
}

// Set some things that should help real-time performance
void realtime_init() {
   struct sched_param sp;
   cpu_set_t cpuset;
   int cpu = 0;

   CPU_ZERO(&cpuset);
   CPU_SET(cpu, &cpuset);
   sched_setaffinity(0, sizeof(cpu_set_t), &cpuset);
   printf("Program tied to CPU#%d\n", cpu);

   sp.sched_priority = 50; //sched_get_priority_max(SCHED_RR); however, 99 not recommended!
   if (sched_setscheduler(0, SCHED_RR, &sp) < 0) {
       printf("Warning: could not change to Round-Robin scheduler\n");
   } else {
       printf("Changed to SCHED_RR with priority %d\n", sp.sched_priority); 
   }

   if (mlockall(MCL_FUTURE | MCL_CURRENT)) {
       perror("Warning: mlockall() failed (not running as root?) with error");
   } else {
       printf("Locked future allocs with mlockall()\n"); 
   }

   mallopt(M_TRIM_THRESHOLD, -1);
   mallopt(M_MMAP_MAX, 0);
   fflush(stdout);
}
