/*
 * lsattr.c		- List file attributes on an ext2 file system
 *
 * Copyright (C) 1993, 1994  Remy Card <card@masi.ibp.fr>
 *                           Laboratoire MASI, Institut Blaise Pascal
 *                           Universite Pierre et Marie Curie (Paris VI)
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/10/30	- Creation
 * 93/11/13	- Replace stat() calls by lstat() to avoid loops
 * 94/02/27	- Integrated in Ted's distribution
 *
 * 95/02/17	- Completly rewriten to display compression information
 *		  and to behave much more like ls. Useless calls to
 *		  intermediate routines that just call ioctl() were
 *		  replaced by direct calls to ioctl(). Uses less space
 *		  in the stack to store the path name when recursive
 *		  listing. Alphabetic listing thanks to scandir().
 *
 *		  	Antoine de Maricourt.
 */

/* Fixme: As for chattr.c, I don't think we should use #ifdef's.  See chattr.c
   for reasons. */

#include <dirent.h>
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#include <fcntl.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
extern int optind;
extern char *optarg;
#endif
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>

#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <linux/ext2_fs.h>

#include "e2p/e2p.h"  /* print_flags() */

const char * program_name = "lsattr";

static int dirs_opt = 0;
static int long_opt = 0;
static int recursive = 0;
static int version_opt = 0;
static int ratio_opt = 0;
static char path[MAXPATHLEN];

/*
 *	Display attributes for one file.
 */

static void
lsattr (char const *name)
{
  int fd;
  char const *msg;

  if ((fd = open (path, O_RDONLY)) == -1)
    fprintf (stderr, "%s: %s: can't open file (%s)\n", program_name, path, strerror (errno));
  else
    {
      unsigned long flags;
      unsigned long version;
#ifdef EXT2_COMPRESS
      unsigned long size;
      unsigned long method	= -1;
      unsigned long ratio[2];
      int unpatched_kernel = 0;
#endif

      if (ioctl (fd, EXT2_IOC_GETFLAGS      , &flags)   == -1 ||
	  ioctl (fd, EXT2_IOC_GETVERSION    , &version) == -1)
	{
	  msg = "can't read flags";
	  goto error;
	}

#ifdef EXT2_COMPRESS
      /* Antoine had what for a while looked like a good idea: if we're
	 running under a kernel not patched for e2compression, then extract
	 the compression method and cluster size from the flag field.
	 Unfortunately, ioctl(EXT2_IOC_GETFLAGS) only gives the low 24 bits of
	 what's on disk.  Hence #if 0. */

# if 0
      /*
       *	In this case, we have a version of lsattr
       *	compiled on a patched system, but running on 
       *	an unpatched one. We extract the method and
       *	the cluster size from the flag field ...!
       */

      /* TODO: On ferait mieux de trouver un autre critere et
	 de regarder errno pour faire un diagnostic plus fin. */
    
      if (ioctl (fd, EXT2_IOC_GETCLUSTERSIZE, &size) == -1 ||
	  ioctl (fd, EXT2_IOC_GETCOMPRMETHOD, &method) == -1)
	{
	  size   = 1 << (((flags >> 24) & 0x03) + 2);
	  method = (flags >> 26) & 0x3f;

	  ratio_opt  = 0;
	}
# else
      ratio_opt &= unpatched_kernel =
	(ioctl (fd, EXT2_IOC_GETCLUSTERSIZE, &size) == -1 ||
	 ioctl (fd, EXT2_IOC_GETCOMPRMETHOD, &method) == -1);
      /* Granted there are other reasons that these ioctls could fail.
	 TODO: Improve this. */
# endif

      if (ratio_opt && ioctl (fd, EXT2_IOC_GETCOMPRRATIO, ratio) == -1)
	{
	  msg = "can't compute ratio";
	  goto error;
	}
#endif

      /*
       *	Version
       */

      if (version_opt)
	printf ("%5lu ", version);

      /*
       *	Flags
       */

      print_flags (stdout, flags, long_opt);

#ifdef EXT2_COMPRESS
      /*
       *	Cluster size
       */
	
      if (unpatched_kernel)
	printf ("  ?");
      else if ((flags & EXT2_COMPR_FL) || (flags & EXT2_COMPRBLK_FL))
	printf (" %2lu", size);
      else
	printf ("  -");

      /*
       *	Compression method
       */

      if (unpatched_kernel)
	printf (" ?      ");
      else if ((flags & EXT2_COMPR_FL))
	switch (method) {
	case EXT2_LZV1_ID	: printf (" lzv1   "); break;
	case EXT2_LZV2_ID	: printf (" lzv2   "); break;
	case EXT2_LZRW3A_ID	: printf (" lzrw3a "); break;
	case EXT2_GZIP1_ID	: printf (" gzip1  "); break;
	case EXT2_GZIP2_ID	: printf (" gzip2  "); break;
	case EXT2_GZIP3_ID	: printf (" gzip3  "); break;
	case EXT2_GZIP4_ID	: printf (" gzip4  "); break;
	case EXT2_GZIP5_ID	: printf (" gzip5  "); break;
	case EXT2_GZIP6_ID	: printf (" gzip6  "); break;
	case EXT2_GZIP7_ID	: printf (" gzip7  "); break;
	case EXT2_GZIP8_ID	: printf (" gzip8  "); break;
	case EXT2_GZIP9_ID	: printf (" gzip9  "); break;
	default:
	  printf (" ?      "); 
	}
      else
	printf (" -      ");

      if (ratio_opt)
	printf (" %5.1f", ratio[1] == 0 ? 100.0 : 100.0 * (double) ratio[1] / ratio[0]);
#endif

      printf (" %s\n", name);

      goto done;

    error:
      fprintf (stderr, "%s: %s: %s (%s)\n", program_name, path, msg, strerror (errno));

    done:
      close (fd);
    }
}

/*
 *	Used by scandir to select file names.
 */

static int
select_normal (const struct dirent *d)
{
  return (d->d_name[0] != '.');
}

static int
select_almost_all (const struct dirent *d)
{
  return (d->d_name[0] != '.' ||
	  (d->d_name[1] != '\0' && (d->d_name[1] != '.' || d->d_name[2] != '\0')));
}

#define select_all NULL

static int (*select_scope) (const struct dirent *) = select_normal;

/*
 *	Show attributes for every file in a directory,
 *	and go deeper if recursive.  Files are
 *	sorted in alphabetical order.
 */

static void
lsattr_dir (int len, int echo)
{
  struct stat st;

  if (len > 0 && lstat (path, &st) == -1)
    {
      fprintf (stderr, "%s: %s: can't stat file (%s)\n", program_name, path, strerror (errno));
      return;
    }

  if (len == 0 || S_ISDIR (st.st_mode))
    {
      struct dirent **lst;
      int i, count;

      if (echo)
	printf ("\n%s:\n", path);

      /*
       *	Scan the directory and sort files in alphabetic order.
       */

      if ((count = scandir (len == 0 ? "." : path, &lst, select_scope, alphasort)) < 0)
	{
	  fprintf (stderr, "%s: %s: can't scan directory (%s)\n",
		   program_name, len == 0 ? "." : path, strerror (errno));
	  return;
	}

      /*
       *	Do files, then subdirectories, as for ls.
       */

      if (len > 0)
	{
	  path[len++] = '/';
	  path[len] = '\0';
	}

      for (i = 0; i < count; i++)
	{
	  strcpy (path + len, lst[i]->d_name);
	  lsattr (lst[i]->d_name);
	}

      if (recursive)
	for (i = 0; i < count; i++)
	  if (select_scope != select_all
	      || lst[i]->d_name[0] != '.'
	      || (lst[i]->d_name[1] != '\0' && (lst[i]->d_name[1] != '.'
						|| lst[i]->d_name[2] != '\0')))
	    {
	      strcpy (path + len, lst[i]->d_name);
	      lsattr_dir (len + strlen(lst[i]->d_name), 1);
	    }

      /*
       *	Free the list.
       */

      for (i = 0; i < count; i++)
	free (lst[i]);

      free (lst);
    }
}

/*
 *	Main routine
 */

/* The behaviour of -dR is the same as GNU ls, i.e. the same as -d. */

static void volatile
usage (void)
{
  fprintf (stderr, 
	   "usage: %s [-ARadhrv] [files ...]\n"
	   "  -A         show almost every file\n"
	   "  -R         recursive mode\n"
	   "  -a         show every file\n"
	   "  -d         directories behave like normal files\n"
	   "  -h         show this help message\n"
#ifdef EXT2_COMPRESS
	   "  -r         show the compression ratio\n"
#endif
	   "  -v         display version number\n",
	   program_name);

  exit (1);
}

int 
main (int argc, char ** argv)
{
  char c;
  int i;

  if (argc && *argv)
    program_name = *argv;

  while ((c = getopt (argc, argv, "ARadhlrv")) != EOF)
    switch (c)
      {
      case 'R' : recursive = 1;	break;
      case 'a' : select_scope = select_all;	break;
      case 'A' : select_scope = select_almost_all;	break;
      case 'd' : dirs_opt = 1;	break;
      case 'l' : long_opt = 1;  	break;
      case 'v' : version_opt = 1;	break;
#ifdef EXT2_COMPRESS
      case 'r' : ratio_opt = 1; 	break;
#endif
      case 'h' :
      default  : usage ();
      }

  path[0] = '\0';

  /* Antoine was thinking that `lsattr .' shouldn't prefix everything with
     `./'.  I don't know what he intended to happen with `lsattr foo . bar'.
     The current behaviour is that with `lsattr' (no file argument), no prefix
     is given; but with `lsattr .' and `lsattr foo . bar', `.' is treated as
     any other directory name.  I think this is the right behaviour as it is,
     so I haven't changed it. */

  if (optind >= argc)
    {
      if (dirs_opt)
	{
	  strcpy (path, ".");
	  lsattr (".");
	}
      else
	lsattr_dir (0, 0);
    }
  else
    for (i = optind; i < argc; i++)
      {
	strcpy (path, argv[i]);
	if (dirs_opt)
	  lsattr (argv[i]);
	else
	  {
	    struct stat st;

	    if (lstat (argv[i], &st) == -1)
	      fprintf (stderr, "%s: %s: can't stat file (%s)\n",
		       program_name, argv[i], strerror (errno));
	    else if (S_ISDIR (st.st_mode)) 
	      lsattr_dir (strlen (argv[i]), 0);
	    else
	      lsattr (argv[i]);
	  }
      }
  return 0;
}
