/*
 * chattr.c		- Change file attributes on an ext2 file system
 *
 * Copyright (C) 1993, 1994  Remy Card <card@masi.ibp.fr>
 *                           Laboratoire MASI, Institut Blaise Pascal
 *                           Universite Pierre et Marie Curie (Paris VI)
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/10/30	- Creation
 * 93/11/13	- Replace stat() calls by lstat() to avoid loops
 * 94/02/27	- Integrated in Ted's distribution
 * 96/09/29	- Integrated Antoine de Maricourt's e2compr changes.
 */

#include <sys/types.h>
#include <dirent.h>
#include <fcntl.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#include <sys/param.h>
#include <sys/stat.h>
#include <linux/ext2_fs.h>

#include "et/com_err.h"
#include "e2p/e2p.h"

#include "../version.h"

const char * program_name = "chattr";

int add = 0;
int rem = 0;
int set = 0;
int set_version = 0;
unsigned long cluster_size = 0;
unsigned long compr_meth   = -1;

unsigned long version;

int recursive = 0;
int verbose = 0;

unsigned long af;
unsigned long rf;
unsigned long sf;

static void volatile fatal_error (const char * fmt_string, int errcode)
{
	fprintf (stderr, fmt_string, program_name);
	exit (errcode);
}

/* help() shows a long usage message and calls exit(0).
   usage() shows a one-liner and calls exit(1). */

#define USAGE_STRING \
	"usage: %s " \
	"[-ERV] [-+=acdisuASX] " \
	"[-b size] [-m name] " \
	"[-v version] files...\n"

#define usage() fatal_error (USAGE_STRING, 1)

static void help (void)
{
	fprintf (stderr,
		 USAGE_STRING
#if EXT2_APPEND_FL
		 "  +-=a       set/reset append-only flag\n"
#endif
		 "  +-=c       set/reset compress flag\n"
#if EXT2_NODUMP_FL
		 "  +-=d       set/reset no-dump flag\n"
#endif
#if EXT2_IMMUTABLE_FL
		 "  +-=i       set/reset immutable flag\n"
#endif
		 "  +-=s       set/reset secure deletion flag\n"
		 "  +-=u       set/reset undelete flag\n"
#if EXT2_NOATIME_FL
		 "  +-=A       set/reset no-atime flag\n"
#endif
		 "  +-=S       set/reset sync flag\n"
#if EXT2_NOCOMPR_FL
		 "  +-=X       enable or disable compression on I/O\n"
#endif
#ifdef EXT2_COMPRESS
		 "  -b size    set cluster size (in blocks)\n"
		 "  -m name    set compression method\n"
#endif
		 "  -h         show this help message\n"
		 "  -v number  set file version\n"
#ifdef EXT2_COMPRESS
		 "  -E         reset compression error flag\n"
#endif
		 "  -R         recursive mode\n"
		 "  -V         verbosely describe changes\n",
		 program_name);

	exit (0);
}

#ifdef EXT2_COMPRESS

/*
 *	Some predefined algorithm names.
 */

static struct { char const *name; int id; } name2id[] = {
  { "lzv"	, EXT2_LZV1_ID	 	},
  { "lzv1"	, EXT2_LZV1_ID	 	},
  { "lzrw3a"	, EXT2_LZRW3A_ID	},
  { "gzip"	, EXT2_GZIP6_ID	 	},
  { "gzip1"	, EXT2_GZIP1_ID	 	},
  { "gzip2"	, EXT2_GZIP2_ID	 	},
  { "gzip3"	, EXT2_GZIP3_ID	 	},
  { "gzip4"	, EXT2_GZIP4_ID	 	},
  { "gzip5"	, EXT2_GZIP5_ID	 	},
  { "gzip6"	, EXT2_GZIP6_ID	 	},
  { "gzip7"	, EXT2_GZIP7_ID	 	},
  { "gzip8"	, EXT2_GZIP8_ID	 	},
  { "gzip9"	, EXT2_GZIP9_ID	 	},
  { NULL }
};
#endif

static int decode_arg (int * i, int argc, char ** argv)
{
	char * p;
	char * tmp;

	switch (argv[*i][0])
	{
	case '-':
		for (p = &argv[*i][1]; *p; p++)
			switch (*p)
			{
#ifdef EXT2_NOATIME_FL
			case 'A':
				rf |= EXT2_NOATIME_FL;
				rem = 1;
				break;
#endif
			case 'E': 
				rf |= EXT2_ECOMPR_FL;
                                rem = 1;
                                break;
			case 'R':
				recursive = 1;
				break;
			case 'S':
				rf |= EXT2_SYNC_FL;
				rem = 1;
				break;
			case 'V':
				verbose = 1;
				break;
#ifdef EXT2_NOCOMPR_FL
                        case 'X':
                        	rf |= EXT2_NOCOMPR_FL;
                                rem = 1;
                                break;
#endif
#ifdef	EXT2_APPEND_FL
			case 'a':
				rf |= EXT2_APPEND_FL;
				rem = 1;
				break;
#endif
#ifdef EXT2_COMPRESS
			case 'b' : 
				(*i)++;
				if (*i >= argc)
					usage ();
				cluster_size = strtol (argv[*i], &tmp, 0);

				if (*tmp || (cluster_size !=  4 &&
					     cluster_size !=  8 &&
					     cluster_size != 16 &&
					     cluster_size != 32))
				{
					com_err (program_name, 0,
						 "invalid cluster size: %s\n",
						 argv[*i]);
					usage ();
				}
				break;

			case 'm' :
			{
				int x;

				(*i)++;
				if (*i >= argc)
					usage ();
				for (x = 0; name2id[x].name != NULL; x++)
					if (strcasecmp (argv[*i], name2id[x].name) == 0)
						break;
	
				if (name2id[x].name == NULL)
				{
					com_err (program_name, 0,
						 "unknown compression method: %s\n",
						 argv[*i]);
					exit (1);
				}

				compr_meth = name2id[x].id;
			}
			break;
#endif
			case 'c':
				rf |= EXT2_COMPR_FL;
				rem = 1;
				break;
#ifdef	EXT2_NODUMP_FL
			case 'd':
				rf |= EXT2_NODUMP_FL;
				rem = 1;
				break;
#endif
#ifdef	EXT2_IMMUTABLE_FL
			case 'h':
				help ();
				break;
			case 'i':
				rf |= EXT2_IMMUTABLE_FL;
				rem = 1;
				break;
#endif
			case 's':
				rf |= EXT2_SECRM_FL;
				rem = 1;
				break;
			case 'u':
				rf |= EXT2_UNRM_FL;
				rem = 1;
				break;
			case 'v':
				(*i)++;
				if (*i >= argc)
					usage ();
				version = strtol (argv[*i], &tmp, 0);
				if (*tmp)
				{
					com_err (program_name, 0,
						 "bad version - %s\n", argv[*i]);
					usage ();
				}
				set_version = 1;
				break;
			default:
				fprintf (stderr, "%s: Unrecognized argument: %c\n",
					 program_name, *p);
				usage ();
			}
		break;
	case '+':
		add = 1;
		for (p = &argv[*i][1]; *p; p++)
			switch (*p)
			{
#ifdef EXT2_NOATIME_FL
			case 'A':
				af |= EXT2_NOATIME_FL;
				break;
#endif
			case 'S':
				af |= EXT2_SYNC_FL;
				break;
#ifdef EXT2_NOCOMPR_FL
                        case 'X':
                        	af |= EXT2_NOCOMPR_FL;
                                break;
#endif
#ifdef	EXT2_APPEND_FL
			case 'a':
				af |= EXT2_APPEND_FL;
				break;
#endif
			case 'c':
				af |= EXT2_COMPR_FL;
				break;
#ifdef	EXT2_NODUMP_FL
			case 'd':
				af |= EXT2_NODUMP_FL;
				break;
#endif
#ifdef	EXT2_IMMUTABLE_FL
			case 'i':
				af |= EXT2_IMMUTABLE_FL;
				break;
#endif
			case 's':
				af |= EXT2_SECRM_FL;
				break;
			case 'u':
				af |= EXT2_UNRM_FL;
				break;
			default:
				usage ();
			}
		break;
	case '=':
		set = 1;
		for (p = &argv[*i][1]; *p; p++)
			switch (*p)
			{
#ifdef EXT2_NOATIME_FL
			case 'A':
				sf |= EXT2_NOATIME_FL;
				break;
#endif
			case 'S':
				sf |= EXT2_SYNC_FL;
				break;
#ifdef EXT2_NOCOMPR_FL
                        case 'X': 
				sf |= EXT2_NOCOMPR_FL;
                                break;
#endif
#ifdef	EXT2_APPEND_FL
			case 'a':
				sf |= EXT2_APPEND_FL;
				break;
#endif
			case 'c':
				sf |= EXT2_COMPR_FL;
				break;
#ifdef	EXT2_NODUMP_FL
			case 'd':
				sf |= EXT2_NODUMP_FL;
				break;
#endif
#ifdef	EXT2_IMMUTABLE_FL
			case 'i':
				sf |= EXT2_IMMUTABLE_FL;
				break;
#endif
			case 's':
				sf |= EXT2_SECRM_FL;
				break;
			case 'u':
				sf |= EXT2_UNRM_FL;
				break;
			default:
				usage ();
			}
		break;
	default:
		return EOF;
		break;
	}
	return 1;
}

static int chattr_dir_proc (const char *, struct dirent *, void *);

static void change_attributes (const char * name)
{
	unsigned long flags;
	struct stat st;
	int fd;

	if (lstat (name, &st) == -1)
	{
		com_err (program_name, errno, "while stat'ing %s", name);
		return;
	}

	/* To the Reader:
	   
	   The problem is that we mustn't close the file between turning on
	   the `c' attribute and setting the compression method or cluster
	   size.  Thus we can't use fsetflags.  I'm in discussion with Remy
	   Card as to the `correct' implementation of this routine (e.g. use
	   of chflags(), and what to do with files that aren't regular files
	   and aren't directories); in the meantime, I suggest using the
	   chattr based on Antoine's code. */
#if 1 /* Untested code for e2compression. */
	fd = open (name, O_RDONLY | O_NONBLOCK);
	if (fd == -1)
	{
		com_err (program_name, errno,
			 "while opening %s", name);
	}
	else
	{
		if (set)
			flags = sf;
		else if (add || rem)
		{
			if (fgetflags (name, &flags) == -1)
			{
				com_err (program_name, errno,
					 "while reading flags on %s", name);
				goto endflags;
			}
			flags &= ~rf;
			flags |= af;
		}
		if (set || add || rem)
		{
			if (verbose)
			{
				printf ("Flags of %s set as ", name);
				print_flags (stdout, sf, 0);
				printf ("\n");
			}
			if (ioctl (fd, EXT2_IOC_SETFLAGS, &flags) == -1)
			{
				if (errno == ENOTTY)
					com_err (program_name, 0,
						 "Invalid request while setting flags on %s", name);
				else
					com_err (program_name, errno,
						 "while setting flags on %s", name);
				goto endflags;
			}
		}

# ifdef EXT2_COMPRESS
		if ((cluster_size != 0) || (compr_meth != -1))
		{
#  if HAVE_EXT2_IOCTLS
			if (cluster_size
			    && (ioctl (fd, EXT2_IOC_SETCLUSTERSIZE,
				       &cluster_size) == -1))
				com_err (program_name, 0,
					 "can't set cluster size of %s", name);

			if ((compr_meth != -1)
			    && (ioctl (fd, EXT2_IOC_SETCOMPRMETHOD,
				       &compr_meth) == -1))
				com_err (program_name, 0,
					 "can't set compression method of %s", name);

#  else /* ! HAVE_EXT2_IOCTLS */
			com_err (program_name, EOPNOTSUPP, "while setting compression parameters");
#  endif /* ! HAVE_EXT2_IOCTLS */
		}
# endif /* EXT2_COMPRESS */

	endflags:
		close (fd);
	}
#else /* End experimental code for e2compression. */

	if (set)
	{
		if (verbose)
		{
			printf ("Flags of %s set as ", name);
			print_flags (stdout, sf, 0);
			printf ("\n");
		}
		if (fsetflags (name, sf) == -1)
			perror (name);
	}
	else
	{
		if (fgetflags (name, &flags) == -1)
			com_err (program_name, errno,
			         "while reading flags on %s", name);
		else
		{
			if (rem)
				flags &= ~rf;
			if (add)
				flags |= af;
			if (verbose)
			{
				printf ("Flags of %s set as ", name);
				print_flags (stdout, flags, 0);
				printf ("\n");
			}
			if (fsetflags (name, flags) == -1)
				com_err (program_name, errno,
				         "while setting flags on %s", name);
		}
	}
#endif /* !experimental */

	if (set_version)
	{
		if (verbose)
			printf ("Version of %s set as %lu\n", name, version);
		if (fsetversion (name, version) == -1)
			com_err (program_name, errno,
			         "while setting version on %s", name);
	}

	if (S_ISDIR(st.st_mode) && recursive)
		iterate_on_dir (name, chattr_dir_proc, (void *) NULL);
}

static int chattr_dir_proc (const char * dir_name, struct dirent * de, void * private)
{
	if (strcmp (de->d_name, ".") && strcmp (de->d_name, ".."))
	{
	        char *path;

		path = malloc(strlen (dir_name) + 1 + strlen (de->d_name) + 1);
		if (!path)
			fatal_error("Couldn't allocate path variable "
				    "in chattr_dir_proc", 1);
		sprintf (path, "%s/%s", dir_name, de->d_name);
		change_attributes (path);
		free(path);
	}
	return 0;
}

void main (int argc, char ** argv)
{
	int i, j;
	int end_arg = 0;

	fprintf (stderr, "chattr %s, %s for EXT2 FS %s, %s\n",
		 E2FSPROGS_VERSION, E2FSPROGS_DATE,
		 EXT2FS_VERSION, EXT2FS_DATE);
	if (argc && *argv)
		program_name = *argv;
	i = 1;
	while (i < argc && !end_arg)
	{
		if (decode_arg (&i, argc, argv) == EOF)
			end_arg = 1;
		else
			i++;
	}
	if (i >= argc)
		usage ();
	if (set && (add || rem))
	{
		fprintf (stderr, "= is incompatible with - and +\n");
		exit (1);
	}
	if ((rf & af) != 0)
	{
		fprintf (stderr, "Can't both set and unset same flag.\n");
		exit (1);
	}
	if (!(add || rem || set || set_version || cluster_size || (compr_meth != -1)))
	{
		fprintf (stderr, "Must use '-b', '-m', '-v', =, - or +\n");
		exit (1);
	}
	for (j = i; j < argc; j++)
		change_attributes (argv[j]);
}
