/*
 *	e2compress.c	Simple program to compress files in a format
 *			transparently readable by the kernel. This
 *			program turns on the EXT2_NOCOMPR_FL flag in
 *			order to have direct access to raw data, and
 *			turns it off again when everything is done.
 *
 *	Copyright (C) 1995, Antoine Dumesnil de Maricourt.
 *
 *	This file can be redistributed under the terms of the GNU General
 *	Public License.
 */

/*
 *	TODO: Behave as chattr for directories.
 *	TODO: Clean termination (remove the tmp file !)
 *	TODO: Catch signals !
 *	TODO: A version that works even on an unpatched kernel
 *	      as lsattr do (possible only for decompression
 *	      because we can guess if a cluster is compressed
 *	      just by looking at the head, for compression we
 *	      have no way to set the cluster bit).
 */

/*
 *	Define this if you want a benchmark program.
 */

#undef BENCH

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <getopt.h>
#include <stdlib.h>

#include <sys/ioctl.h>
#include <sys/stat.h>
#ifdef BENCH
#include <sys/time.h>
#include <sys/resource.h>
#endif

#include <linux/ext2_fs.h>

/*
 *	Default parameters
 */

static int	cluster_size	= EXT2_DEFAULT_CLUSTER_SIZE;
static int	compr_meth	= EXT2_DEFAULT_COMPR_METHOD;
static int	do_compress	= 1;
static int	verbose		= 0;
static int	do_continue	= 0;
static int	do_verify	= 0;
static int	fs_blk_size	= 1024;

static const char *prog_name	= "e2compress";
void   *heap_base	= NULL;

static const char   *suffix		= "~z~";

static int	do_bench	= 0;

#ifdef BENCH
int	loop		= 10;

unsigned long cmsec	= 0;
unsigned long umsec	= 0;
unsigned long cbyte	= 0;
unsigned long ubyte	= 0;
unsigned long cblk	= 0;
unsigned long ublk	= 0;
#endif

#define HEAP_SIZE	(256 * 1024)

/*
 *	Defined in ext2_fs.h, but under the protection of __KERNEL__
 */

extern size_t ext2_iLZV1   (void*);
extern size_t ext2_iLZV2   (void*);
extern size_t ext2_iLZRW3A (void*);
extern size_t ext2_iNONE   (void*);
extern size_t ext2_iGZIP   (void*);

extern size_t ext2_wLZV1   (__u8*, __u8*, size_t, size_t, int);
extern size_t ext2_wLZV2   (__u8*, __u8*, size_t, size_t, int);
extern size_t ext2_wLZRW3A (__u8*, __u8*, size_t, size_t, int);
extern size_t ext2_wNONE   (__u8*, __u8*, size_t, size_t, int);
extern size_t ext2_wGZIP   (__u8*, __u8*, size_t, size_t, int);

extern size_t ext2_rLZV1   (const __u8*, __u8*, size_t, size_t, int);
extern size_t ext2_rLZV2   (const __u8*, __u8*, size_t, size_t, int);
extern size_t ext2_rLZRW3A (const __u8*, __u8*, size_t, size_t, int);
extern size_t ext2_rNONE   (const __u8*, __u8*, size_t, size_t, int);
extern size_t ext2_rGZIP   (const __u8*, __u8*, size_t, size_t, int);

extern unsigned long ext2_adler32 (unsigned long, const unsigned char*, int);

/*
 *	Algorithm table.
 */

struct ext2_algorithm { 
  const char    *name; 
  size_t (*init)	(void*); 
  size_t (*compress)	(__u8*, __u8*, size_t, size_t, int); 
  size_t (*uncompress)	(const __u8*, __u8*, size_t, size_t, int);
  int	   xarg;
};

static struct ext2_algorithm algo[32] = {

  { "lzv1"	, ext2_iLZV1	, ext2_wLZV1	, ext2_rLZV1	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},

  { "lzrw3a"	, ext2_iLZRW3A	, ext2_wLZRW3A	, ext2_rLZRW3A	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},

  { "gzip1"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 1	},
  { "gzip2"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 2	},
  { "gzip3"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 3	},
  { "gzip4"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 4	},
  { "gzip5"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 5	},
  { "gzip6"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 6	},
  { "gzip7"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 7	},
  { "gzip8"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 8	},

  { "gzip9"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 9	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
};

/*
 *	Algorithm name to ID conversion (should be shared with chattr)
 */

static struct { const char *name; int id; } name2id[] = {
  { "lzv"	, EXT2_LZV1_ID	 	},
  { "lzv1"	, EXT2_LZV1_ID	 	},
  { "lzrw3a"	, EXT2_LZRW3A_ID	},
  { "gzip"	, EXT2_GZIP6_ID	 	},
  { "gzip1"	, EXT2_GZIP1_ID	 	},
  { "gzip2"	, EXT2_GZIP2_ID	 	},
  { "gzip3"	, EXT2_GZIP3_ID	 	},
  { "gzip4"	, EXT2_GZIP4_ID	 	},
  { "gzip5"	, EXT2_GZIP5_ID	 	},
  { "gzip6"	, EXT2_GZIP6_ID	 	},
  { "gzip7"	, EXT2_GZIP7_ID	 	},
  { "gzip8"	, EXT2_GZIP8_ID	 	},
  { "gzip9"	, EXT2_GZIP9_ID	 	},
  { NULL	, 0 }
};

/*
 *	Help message
 */

static void volatile 
usage (int rc)
{
  fprintf (stderr, "\
Usage: %s [-b size] [-m name] [-F size] [-t] [-v] files ...
 %s -d [-F size] [-v] files ...
 %s -h
  -F size	 supply filesystem blocksize. (Default=1024.)
  -S suffix  set backup suffix (default to '~z~')
  -t         verify compression
  -b size    set cluster size for compression
  -d         decompress files
  -m name    set compression algorithm for compression
  -h         show this help message
  -v         display cluster statistics
",
	   prog_name, prog_name, prog_name);
  exit (rc);
}

#define RoundUpDiv(_n,_s) (((_n) + (_s) - 1) / (_s))

typedef struct file_status {
  const char   *path;
  off_t		ilen;
  unsigned long	iflags;
  int		restore_flags;
  int		ifd;
  int		ofd;
  unsigned long iblk;
  unsigned long cblk;
  int		blk_size;
  int		buf_size;
  int		clu_nblocks;
  unsigned long	clusters;
#ifdef BENCH
  unsigned long	cmsec;
  unsigned long	umsec;
  unsigned long	cbyte;
  unsigned long	ubyte;
  unsigned long	blk[2];
#endif
} file_status;

static int 
fatal_error (file_status const * const status, const char *msg, int err)
{
  unsigned long value;

  if (msg != NULL) {
    fprintf (stderr, "%s: %s: %s", prog_name, status->path, msg);

    if (err != 0)
      fprintf (stderr, " (%s)", strerror (err));

    fprintf (stderr, "\n");
  }

  if (status->ifd != -1) {
    if (status->restore_flags)
      {
	value = status->iflags;
	(void) ioctl (status->ifd, EXT2_IOC_SETFLAGS, &value);
      }

    close (status->ifd);
  }

  if (status->ofd != -1)
    close (status->ofd);

  return -1;
}

/*
 *	Link original file to backup file, create the output file and
 *	turn off compression for it. Cluster size and algorithm are set
 *	for the new file. Various values are also computed to help the
 *	main compression routine. After that, we are ready to copy the
 *	original file into the compressed file.
 */

static int 
open_files (const char *path, file_status *status)
{
  unsigned long value;
  struct stat   st;

  status->path		= path;
  status->restore_flags	= 0;
  status->ifd		= -1;
  status->ofd		= -1;
  status->iblk		= 0;
  status->cblk		= 0;
  status->clu_nblocks	= cluster_size;

#ifdef BENCH
  status->cmsec		= 0;
  status->umsec		= 0;
  status->cbyte		= 0;
  status->ubyte		= 0;
  status->blk[0]	= 0;
  status->blk[1]	= 0;
#endif

  /*
   *	Open input file.
   */

  if ((status->ifd = open (path, O_RDONLY, 0)) == -1)
    return fatal_error (status, "can't open file", errno);

  if (ioctl (status->ifd, EXT2_IOC_GETFLAGS, &value) == -1)
    return fatal_error (status, "can't read flags", errno);
  status->iflags = value;

  if (fstat (status->ifd, &st) == -1)
    return fatal_error (status, "can't stat file", errno);

  if (! S_ISREG (st.st_mode))
    return fatal_error (status, "not a regular file", 0);

  if (st.st_nlink > 1 && ! do_continue)
    return fatal_error (status, "file has more than one link", 0);

  if (! do_compress)
    {
      /*
       *	If we are uncompressing, we turn off automatic compression
       *	for input file, and read the cluster size.
       */

      if (!(status->iflags & (EXT2_COMPR_FL | EXT2_COMPRBLK_FL))
	  && !do_continue)
	return fatal_error (status, "file not compressed", 0);

      if (ioctl (status->ifd, EXT2_IOC_GETCLUSTERSIZE, &value) == -1)
	return fatal_error (status, "can't read cluster size", errno);
      status->clu_nblocks = value;

      value = status->iflags | EXT2_NOCOMPR_FL;
      if (ioctl (status->ifd, EXT2_IOC_SETFLAGS, &value) == -1)
	return fatal_error (status, "can't turn off compression", errno);

      status->restore_flags = 1;
    }

  status->ilen		= st.st_size;
  status->blk_size	= fs_blk_size;  /* _not_ st.st_blksize */
  status->buf_size	= status->clu_nblocks * fs_blk_size;
  status->clusters	= RoundUpDiv (st.st_size, status->buf_size);

  /*
   *	Set up output file.
   */
  {
    char opath[strlen (path) + strlen (suffix) + 1];
    struct stat ost;

    strcpy (opath, path);
    strcat (opath, suffix);

    if (stat (opath, &ost) != -1 && ! do_continue)
      return fatal_error (status, "temporary file already exists", 0);

    if ((status->ofd = open (opath, O_RDWR | O_CREAT, st.st_mode & ~S_IFMT)) == -1)
      return fatal_error (status, "can't open output file", errno);

    if (fstat (status->ofd, &ost) == -1)
      return fatal_error (status, "can't stat temporary file", errno);
    
    if (st.st_dev != ost.st_dev)
      return fatal_error (status, "input and temporary files not on same device", 0);

    if (st.st_ino == ost.st_ino)
      return fatal_error (status, "input and temporary files are the same", 0);

    if (ftruncate (status->ofd, 0) == -1)
      return fatal_error (status, "can't truncate output file", errno);
  }

  /*
   *	Copy original properties (TODO: the time).
   */

  if (fchown (status->ofd, st.st_uid, st.st_gid) == -1 && ! do_continue)
    return fatal_error (status, "can't restore original owner/group", errno);

  if (ftruncate (status->ofd, status->ilen) == -1)
    return fatal_error (status, "can't truncate output file", errno);

  if (do_compress)
    {
      /*
       *	If we are compressing, turn off automatic compression
       *	for the output file.
       */

      value = status->iflags | EXT2_NOCOMPR_FL | EXT2_COMPR_FL;
      if (ioctl (status->ofd, EXT2_IOC_SETFLAGS, &value) == -1)
	return fatal_error (status, "can't turn off compression", errno);

      value = status->clu_nblocks;
      if (ioctl (status->ofd, EXT2_IOC_SETCLUSTERSIZE, &value) == -1)
	return fatal_error (status, "can't set cluster size", errno);

      value = compr_meth;
      if (ioctl (status->ofd, EXT2_IOC_SETCOMPRMETHOD, &value) == -1)
	return fatal_error (status, "can't set compression method", errno);
    }
  else 
    {
      value = status->iflags & ~EXT2_COMPR_FL;
      if (ioctl (status->ofd, EXT2_IOC_SETFLAGS, &value) == -1)
	return fatal_error (status, "can't restore original flags", errno);
    }

  if (verbose)
    fprintf (stderr, "File '%s': %ld bytes, %ld cluster%c\n", 
	     path, status->ilen, status->clusters, status->clusters > 1 ? 's' : ' ');

  return 0;
}


/*
 *	Everything is done, we delete the backup file and close
 *	the output file. Original flags are restored.
 */

static int 
close_files (file_status *status)
{
  unsigned long value;

  if (do_compress) 
    {
      value = (status->iflags | EXT2_COMPR_FL) & ~EXT2_NOCOMPR_FL;

      if (ioctl (status->ofd, EXT2_IOC_SETFLAGS, &value) == -1)
	return fatal_error (status, "can't set flags", errno);
    }
  else
    {
      value = status->iflags; (void) ioctl (status->ifd, EXT2_IOC_SETFLAGS, &value);
    }
    
  {
    char opath[strlen (status->path) + strlen (suffix) + 1];

    strcpy (opath, status->path);
    strcat (opath, suffix);

#ifdef BENCH
    if (do_bench) 
      {
	if (ioctl (status->ofd, EXT2_IOC_GETCOMPRRATIO, &(status->blk)) == -1)
	  return fatal_error (status, "can't get ratio", errno);

	cblk += status->blk[0];
	ublk += status->blk[1];

	unlink (opath);
      }

    else
#endif

      if (rename (opath, status->path) == -1)
	return fatal_error (status, "can't rename file", errno);
  }

#ifdef BENCH
  if (do_bench)
    {
      printf ("%-12s  %7ld  %7ld   %5.1f",
	      status->path, status->ubyte, status->cbyte, 
	      (status->blk[1] != 0 && status->blk[1] != status->blk[0] 
	       ? (double) 100.0 * status->blk[0] / status->blk[1] 
	       : 100.0));
    
      if (status->cmsec > 0)
	printf ("  %5.2f MB/s",
		(double) status->ubyte * loop / (1000.0 * status->cmsec));
      else 
	printf ("   ?.?? MB/s");

      if (status->umsec > 0)
	printf (" %5.2f MB/s",
		(double) status->ubyte * loop / (1000.0 * status->umsec));
      else 
	printf ("  ?.?? MB/s");

      if (status->umsec < 500 || status->cmsec < 500)
	printf ("  *\n"); 
      else 
	printf ("\n");
    }

  cmsec += status->cmsec;
  umsec += status->umsec;
  cbyte += status->cbyte;
  ubyte += status->ubyte;
#endif

  close (status->ifd);
  close (status->ofd);

  return 0;
}


static inline int
mflcv_verify_compression (int const ilen,
			  int const clen,
			  file_status * const status,
			  unsigned char const * const ibuf,
			  unsigned char * const obuf)
{
  int vlen = ilen;
  static unsigned char *vbuf = NULL;
#ifdef BENCH
  struct rusage old, new;
#endif

  /* effic: We can probably share ibuf or obuf. */
  if ((vbuf = realloc (vbuf, status->buf_size)) == NULL)
    return fatal_error (status, "cannot allocate vbuf", 0);

#ifdef BENCH
  getrusage (RUSAGE_SELF, &old);
	  
  for (i = 0; i < loop; i++)
    {
#endif
      if (algo[compr_meth].init (heap_base) > HEAP_SIZE)
	return fatal_error (status, "heap not big enough !!!", 0);

      vlen = algo[compr_meth].uncompress (obuf + sizeof (struct ext2_cluster_head),
					  vbuf,
					  clen,
					  vlen,
					  algo[compr_meth].xarg);

#ifdef BENCH
    }

  getrusage (RUSAGE_SELF, &new);
  status->umsec += ((new.ru_utime.tv_sec - old.ru_utime.tv_sec) * 1000
		    + new.ru_utime.tv_usec - old.ru_utime.tv_usec);
#endif

  if ((vlen <= 0) || (vlen != ilen) || (memcmp (ibuf, vbuf, ilen) != 0))
    return fatal_error (status, "verification failed", 0);

  return 0;
}

static unsigned long
mflch_create_head (int const ulen,
		   int const clen,
		   unsigned long const ubitmap,
		   unsigned char const * const ubuf,
		   unsigned char * const cbuf)
{
  struct ext2_cluster_head *head  = (struct ext2_cluster_head *) cbuf;
  head->magic	    = EXT2_COMPRESS_MAGIC;
  head->len	    = ulen;
  head->compr_len   = clen;
  head->method	    = compr_meth;
  head->reserved_0  = 0;
  head->bitmap	    = ubitmap;
  return head->reserved_2 = ext2_adler32 (1, ubuf, ulen);
}


static inline int
mflc_compress_cluster (unsigned long const cluster,
		       struct file_status * const status,
		       int const ulen,
		       int * const oleft_p,
		       unsigned char const ** const optr_p,
		       unsigned long const ubitmap,
		       unsigned long * const obitmap,
		       unsigned char * const ibuf,
		       unsigned char * const obuf)
{
  int clen;
#ifdef BENCH
  int i;
  struct rusage old;
  struct rusage new;
#endif

  if (ulen <= status->blk_size)
    clen = 0;   /* Won't compress to a smaller number of blocks. */
  else
    {
#ifdef BENCH
      getrusage (RUSAGE_SELF, &old);

      for (i = 0; i < loop; i++)
	{
#endif
	  if (algo[compr_meth].init (heap_base) > HEAP_SIZE)
	    return fatal_error (status, "heap not big enough !!!", 0);

	  clen = algo[compr_meth].compress (ibuf, 
					    obuf + sizeof (struct ext2_cluster_head), 
					    ulen, 
					    (((ulen - 1)
					      & ~(status->blk_size - 1))
					     - sizeof (struct ext2_cluster_head)),
					    algo[compr_meth].xarg);

#ifdef BENCH
	}
	  
      getrusage (RUSAGE_SELF, &new);
	  
      status->cmsec += ((new.ru_utime.tv_sec - old.ru_utime.tv_sec) * 1000
			+ new.ru_utime.tv_usec - old.ru_utime.tv_usec);
#endif
    }

#ifdef BENCH
  status->ubyte += ulen;
  status->cbyte += clen;
#endif

  if (clen == 0)
    {   	/* Couldn't get it any smaller. */
      if (ioctl (status->ofd, EXT2_IOC_CLRCLUSTERBIT, &cluster) == -1)
	return fatal_error (status, "can't clear cluster bit", errno);

      *optr_p = ibuf;
      *oleft_p = ulen;
      *obitmap = ubitmap;

      if (verbose)
	{
	  int nblks = RoundUpDiv (ulen, status->blk_size);
	  fprintf (stderr, " cluster #%3ld, ubitmap=0x%08lx, blocks=%2d/%2d\n",
		   cluster, ubitmap, nblks, nblks);
	}
    }
  else
    {
      unsigned long adler32;

      if (ioctl (status->ofd, EXT2_IOC_SETCLUSTERBIT, &cluster) == -1)
	return fatal_error (status, "can't set cluster bit", errno);

      adler32 = mflch_create_head (ulen, clen, ubitmap, ibuf, obuf);
	  
      if ((!do_continue) || do_verify || do_bench)
	if (mflcv_verify_compression (ulen, clen, status, ibuf, obuf))
	  return -1;

      *optr_p = obuf;
      *oleft_p = clen + sizeof (struct ext2_cluster_head);
      *obitmap = (1 << RoundUpDiv (clen + sizeof (struct ext2_cluster_head), status->blk_size)) - 1;
  
      if (verbose)
	fprintf (stderr, " cluster #%3ld, ubitmap=0x%08lx, blocks=%2d/%2d, adler32=0x%08lx\n",
		 cluster, ubitmap, 
		 RoundUpDiv (clen + sizeof (struct ext2_cluster_head), status->blk_size),
		 RoundUpDiv (ulen, status->blk_size),
		 adler32);
    }

  return 0;
}

static int
mfld_decompress_cluster (unsigned long const cluster,
			 file_status * const status,
			 int const ilen,
			 int * const oleft_p,
			 unsigned char const ** const optr_p,
			 unsigned long const ibitmap,
			 unsigned long * const obitmap,
			 unsigned char const * const ibuf,
			 unsigned char * const obuf)
{
  unsigned long value = cluster;
  struct ext2_cluster_head const *head = (struct ext2_cluster_head const *) ibuf;

  if (ioctl (status->ifd, EXT2_IOC_GETCLUSTERBIT, &value) == -1)
    return fatal_error (status, "can't read cluster bit", errno);

  if (value)
    {
      /* Cluster is compressed. */
      int clen, ulen;

      if (head->magic != EXT2_COMPRESS_MAGIC)
	return fatal_error (status, "wrong magic number", 0);

      if (ibitmap & (ibitmap + 1))
	return fatal_error (status, "Warning! Hole found at unexpected place; please send bug report.", 0);

      *obitmap = head->bitmap;
      clen = head->compr_len;

      if (algo[head->method].init (heap_base) > HEAP_SIZE)
	return fatal_error (status, "heap not big enough !!!", 0);

      ulen = algo[head->method].uncompress (ibuf + sizeof (struct ext2_cluster_head),
					    obuf,
					    clen,
					    head->len,
					    algo[compr_meth].xarg);

      if ((ulen != head->len)
	  || ((head->reserved_2 != 0)
	      && (ext2_adler32 (1, obuf, ulen) != head->reserved_2)))
	return fatal_error (status, "uncompression error", 0);
	
      *optr_p = obuf;
      *oleft_p = ulen;

      if (verbose)
	{
	  fprintf (stderr, " cluster #%3ld, obitmap=0x%08lx, blocks=%2d/%2d, adler=0x%08lx\n",
		   cluster, *obitmap, 
		   RoundUpDiv (clen + sizeof (struct ext2_cluster_head), status->blk_size),
		   RoundUpDiv (ulen, status->blk_size),
		   (unsigned long) head->reserved_2);
	}
    }
  else
    {
      /* Cluster is already stored in uncompressed form. */
      *optr_p = ibuf;
      *oleft_p = ilen;

      if (verbose)
	fprintf (stderr, " cluster #%3ld, obitmap=0x%08lx, wasn't compressed\n",
		 cluster, *obitmap = ibitmap);
    }

  return 0;
}

static inline int
mfl_process_cluster (unsigned long const cluster,
		     struct file_status * const status,
		     unsigned char * const ibuf,
		     unsigned char * const obuf)
{
  int ileft = status->ilen;
  int oleft;
  off_t offset = 0;
  int block;
  unsigned char const *optr;
  unsigned char *iptr = ibuf;
  unsigned long ibitmap = 0;
  unsigned long obitmap;
  int ilen = 0;

  /*
   *	Read the cluster, creating ibitmap as we go.
   */

  for (block = 0; (block < status->clu_nblocks) && (ileft > 0); block++)
    {
      int chars = (ileft <= status->blk_size
		   ? ileft
		   : status->blk_size);

      iptr[0] = iptr[1] = iptr[2] = iptr[3] = 0;
      if (read (status->ifd, iptr, chars) != chars)
	return fatal_error (status, "can't read from input file", errno);

      if ((iptr[0] != 0)
	  || (iptr[1] != 0)
	  || (iptr[2] != 0)
	  || (iptr[3] != 0)
	  || ((chars > 4)
	      && (memcmp (iptr, iptr + 4, chars - 4) != 0)))
	{
	  ibitmap |= 1 << block;
	  iptr += chars;
	  ilen += chars;
	}

      ileft -= chars;
    }

  if (do_compress)
    {
      if (mflc_compress_cluster (cluster, status, ilen, &oleft, &optr, ibitmap, &obitmap, ibuf, obuf))
	return -1;
    }
  else
    {
      if (mfld_decompress_cluster (cluster, status, ilen, &oleft, &optr, ibitmap, &obitmap, ibuf, obuf))
	return -1;
    }

  /*
   *	Write back the cluster.
   */

  for (block = 0; block < status->clu_nblocks; block++)
    {
      if (oleft > 0)
	{
	  int chars = (oleft > status->blk_size
		       ? status->blk_size
		       : oleft);

	  if (obitmap & (1 << block))
	    {
	      if (lseek (status->ofd, offset, SEEK_SET) != offset)
		return fatal_error (status, "can't seek output file", errno);

	      if (write (status->ofd, optr, chars) != chars)
		return fatal_error (status, "can't write to output file", errno);

	      oleft -= chars;
	      optr  += chars;
	    }
	}
	
      offset += status->blk_size;
    }
  return 0;
}


static int 
mf_process_file (const char *path)
{
  struct file_status status;
  unsigned long cluster;
  static unsigned char *ibuf = NULL;
  static unsigned char *obuf = NULL;

  if (open_files (path, &status) != 0)
    return -1;

  if (heap_base == NULL && (heap_base = malloc (HEAP_SIZE)) == NULL)
    return fatal_error (&status, "can't allocate heap", 0);

  if ((ibuf = realloc (ibuf, status.buf_size)) == NULL)
    return fatal_error (&status, "can't allocate buffers", 0);

  if ((obuf = realloc (obuf, status.buf_size)) == NULL)
    return fatal_error (&status, "can't allocate buffers", 0);

  for (cluster = 0; cluster < status.clusters; cluster++)
    if (mfl_process_cluster (cluster, &status, ibuf, obuf))
      return -1;

  close_files (&status);
  return 0; 
}

int 
main (int argc, char *argv[])
{
  int	c, i;
  char *tmp;

  if (argc && *argv)
    prog_name = *argv;

  if (strcmp (prog_name, "e2uncompress") == 0)
    do_compress = 0;

  while ((c = getopt (argc, argv, "m:A:BtF:N:S:b:dfhv")) != EOF)
    switch (c) {
    case 'd': do_compress = 0; break;
    case 'f': do_continue = 1; break;
    case 't': do_verify   = 1; break;
    case 'v': verbose     = 1; break;
#ifdef BENCH
    case 'B': do_bench    = 1; break;
    case 'N': loop = strtol (optarg, &tmp, 0); break;
#endif
    case 'F':
      fs_blk_size = strtol (optarg, &tmp, 0);
      if (*tmp)
	usage (1);
      if ((fs_blk_size & (fs_blk_size - 1))
	  || (fs_blk_size < EXT2_MIN_BLOCK_SIZE)
	  || (fs_blk_size > EXT2_MAX_BLOCK_SIZE))
	{
	  fprintf (stderr,
		   "%s: invalid fs blocksize `%s'.\n"
		   "  Ought to be a power of two in the range ["
		   __STRING(EXT2_MIN_BLOCK_SIZE)
		   ", "
		   __STRING(EXT2_MAX_BLOCK_SIZE)
		   "].\n",
		   prog_name, argv[optind]);
	  exit (1);
	}
      break;

    case 'b':
      cluster_size = strtol (optarg, &tmp, 0);
      if (*tmp || (cluster_size !=  4 && cluster_size !=  8 &&
		   cluster_size != 16 && cluster_size != 32))
	{
	  fprintf (stderr, "%s: invalid cluster size: %s\n",
		   prog_name, argv[optind]);
	  exit (1);
	}
      break;
    
    case 'S':
      suffix = optarg;
      break;

    case 'm':
    case 'A':
      {
	int x;
	
	for (x = 0; name2id[x].name != NULL; x++)
	  if (strcasecmp (optarg, name2id[x].name) == 0)
	    break;
	
	if (name2id[x].name == NULL)
	  {
	    fprintf (stderr, "%s: unknown algorithm: %s\n", prog_name, optarg);
	    exit (1);
	  }

	compr_meth = name2id[x].id;
      }
      break;

    case 'h':
      usage (0);
    default:
      usage (1);
    }

#ifdef BENCH
  if (! do_compress)
    do_bench = 0;
#endif

  if (optind < argc)
    {
#ifdef BENCH
      if (! do_bench)
	loop = 1;
      else
	{
	  printf ("--------------------------------------\n");

	  for (i = 0; i < optind; i++)
	    printf ("%s ", argv[i]); printf ("\n\n");

	    printf ("File name      Length            Ratio\n");
	    printf ("---------      ------            -----\n");
	}
#endif

      for (i = optind; i < argc; i++)
	mf_process_file (argv[i]);

#ifdef BENCH
      if (do_bench)
	{
	  printf ("---------      ------            -----\n");
	  printf ("%-7s [%2d]  %7ld  %7ld   %5.1f",
		  algo[compr_meth].name, cluster_size, ubyte, cbyte, 
		  ((ublk != 0) && (cblk != ublk)
		   ? (double) 100.0 * cblk / ublk
		   : 100.0));
    
	  if (cmsec > 0)
	    printf ("  %5.2f MB/s", (double) ubyte * loop / (1000.0 * cmsec));
	  else
	    printf ("  ??.?? MB/s");
      
	  if (umsec > 0)
	    printf (" %5.2f MB/s\n", (double) ubyte * loop / (1000.0 * umsec));
	  else
	    printf ("  ??.?? MB/s\n");
	}
#endif
    }

  else 
    usage (1);

  return 0;
}
