/* Work in progress:

   FIXME: I had assumed that compressed data is stored in contiguous
   blocks at the beginning of each cluster.  Actually, this is not
   correct.  The compressed data go in a set of blocks which is a
   subset of the blocks occupied by the uncompressed data.  The
   current code will fail if there is a hole near the beginning of the
   cluster.

   Make decompression work for stdin and non-ext2
   filesystems.  make iflags read ioctl not cause abort on failure.

   Look at the compression routines.

   Probably other things.

   */


/* e2compress2.c: works on unpatched kernels.  Once this is finished,
   it will be renamed `e2compress.c'. */


/*
    e2compress.c	Simple program to compress files in a format
			transparently readable by the kernel.  This
  			program turns on the EXT2_NOCOMPR_FL flag in
  			order to have direct access to raw data, and
  			turns it off again when everything is done.
  
  
    The idea and portions of the current code are copyright (C) 1995,
    Antoine Dumesnil de Maricourt.
  
    The remainder is copyright (C) 1996-7, Peter Moulder.  See file
    `CREDITS' for separation, but it is probably unimportant given
    that both Peter and Antoine have placed their work under the GNU
    General Public License.
  

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


    Peter can be contacted at <reiter@netspace.net.au> or
    1/36 Auburn Grove, Hawthorn East 3123, Australia.

    Antoine is no longer the maintainer of the e2compr project so
    should not normally be contacted for maintenance issues (bugs,
    feature requests, etc.), but his e-mail address is
    <dumesnil@etca.fr>.

 */

/*
 * TODO: Behave as chattr for directories.
 *
 * TODO: Clean termination (remove the tmp file !)
 *
 * TODO: Catch signals !
 *
 * TODO: Allow decompressing stdin.  (Set .path to "(stdin)" for diagnostics.)
 *
 * TODO: We currently assume that if getflags fails then file is not
 *	 on an ext2 fs.  Check the kernel ioctl code to see if this is
 *	 true.
 * todo: separate --force.  Warn if compress compressed file.  (Skip unless -f.)
 */

/* Define this if you want to enable support for benchmarking.  */
/* Note: at present, the benchmarking code is unusable.  (TODO.) */
#undef BENCH  /* I'll reenable this once benchmarking stuff is worked out. */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <getopt.h>
#include <stdlib.h>
#include <sys/types.h>
#include <utime.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/mman.h>	/* mmap */
#include <values.h>	/* BITSPERBYTE */
#ifdef BENCH
#include <sys/time.h>
#include <sys/resource.h>
#endif
#include <assert.h>
#include <linux/ext2_fs.h>

typedef enum { false, true } bool;


static char const version_string[] = "e2compress 1997-01-14";
static const char *prog_name;
void *heap_base = NULL;


#ifdef BENCH
static unsigned long cmsec	= 0;
static unsigned long umsec	= 0;
static unsigned long cbyte	= 0;
static unsigned long ubyte	= 0;
static unsigned long cblk	= 0;
static unsigned long ublk	= 0;
#endif

#define HEAP_SIZE	(256 * 1024)


/*
 *	Defined in ext2_fs.h, but under the protection of __KERNEL__
 */

extern size_t ext2_iLZV1   (void*);
extern size_t ext2_iLZV2   (void*);
extern size_t ext2_iLZRW3A (void*);
extern size_t ext2_iNONE   (void*);
extern size_t ext2_iGZIP   (void*);

extern size_t ext2_wLZV1   (__u8*, __u8*, size_t, size_t, int);
extern size_t ext2_wLZV2   (__u8*, __u8*, size_t, size_t, int);
extern size_t ext2_wLZRW3A (__u8*, __u8*, size_t, size_t, int);
extern size_t ext2_wNONE   (__u8*, __u8*, size_t, size_t, int);
extern size_t ext2_wGZIP   (__u8*, __u8*, size_t, size_t, int);

extern size_t ext2_rLZV1   (const __u8*, __u8*, size_t, size_t, int);
extern size_t ext2_rLZV2   (const __u8*, __u8*, size_t, size_t, int);
extern size_t ext2_rLZRW3A (const __u8*, __u8*, size_t, size_t, int);
extern size_t ext2_rNONE   (const __u8*, __u8*, size_t, size_t, int);
extern size_t ext2_rGZIP   (const __u8*, __u8*, size_t, size_t, int);

extern unsigned long ext2_adler32 (unsigned long adler, const unsigned char *buf, unsigned len);


/*
 *	Algorithm table.
 */

struct ext2_algorithm { 
  const char    *name; 
  size_t (*init)	(void*); 
  size_t (*compress)	(__u8*, __u8*, size_t, size_t, int); 
  size_t (*uncompress)	(const __u8*, __u8*, size_t, size_t, int);
  int	   xarg;
};

static struct ext2_algorithm algo[32] = {

  { "lzv1"	, ext2_iLZV1	, ext2_wLZV1	, ext2_rLZV1	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},

  { "lzrw3a"	, ext2_iLZRW3A	, ext2_wLZRW3A	, ext2_rLZRW3A	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},

  { "gzip1"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 1	},
  { "gzip2"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 2	},
  { "gzip3"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 3	},
  { "gzip4"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 4	},
  { "gzip5"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 5	},
  { "gzip6"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 6	},
  { "gzip7"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 7	},
  { "gzip8"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 8	},

  { "gzip9"	, ext2_iGZIP	, ext2_wGZIP	, ext2_rGZIP	, 9	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
  { NULL	, ext2_iNONE	, ext2_wNONE	, ext2_rNONE	, 0	},
};

/*
 *	Algorithm name to ID conversion (should be shared with chattr)
 */

static struct { const char *name; int id; } name2id[] = {
  { "lzv"	, EXT2_LZV1_ID	 	},
  { "lzv1"	, EXT2_LZV1_ID	 	},
  { "lzrw3a"	, EXT2_LZRW3A_ID	},
  { "gzip"	, EXT2_GZIP6_ID	 	},
  { "gzip1"	, EXT2_GZIP1_ID	 	},
  { "gzip2"	, EXT2_GZIP2_ID	 	},
  { "gzip3"	, EXT2_GZIP3_ID	 	},
  { "gzip4"	, EXT2_GZIP4_ID	 	},
  { "gzip5"	, EXT2_GZIP5_ID	 	},
  { "gzip6"	, EXT2_GZIP6_ID	 	},
  { "gzip7"	, EXT2_GZIP7_ID	 	},
  { "gzip8"	, EXT2_GZIP8_ID	 	},
  { "gzip9"	, EXT2_GZIP9_ID	 	},
  { NULL	, 0 }
};

/*
 *	Help message
 */

/* Todo: Update this usage text to show long options.  Also modify -B interface. */
static void volatile 
usage (int rc)
{
  fprintf (stderr, "\
Usage: %s [-b size] [-m method] [-F size] [-f] [-t] [-v] files ...
  %s -d [-F size] [-f] [-v] files ...
  %s -h | --help | --version
Options:
  -F, --fs-blocksize <size>
	Specify that the filesystem blocksize = <size>.  (Default=1024.)
  -S, --suffix <suffix>
	Set backup suffix to <suffix>.  (Default=`~z~')
"
#ifdef BENCH
	   "\
  -B --benchmark
	Provide timing information for benchmarking.
  -N --loops <count>
	Set the number of times to loop with -B (default=10).
"
#endif
	   "\
  -f, --force 
	Force: 
	 -  write over existing backup file, 
	 -  continue even if more than one link (the other
	    links will remain unchanged),
	 -  attempt to uncompress even if original isn't flagged as
	    being compressed,
	 -  ignore errors in copying original owner/group,
	 -  disables verification of compression
  -b, --cluster-blocks <size>
	Set cluster size (number of filesystem blocks) for compression.
  -d, --decompress, --uncompress
	Decompress files.
  -m, --method, --algorithm <method>
	Set compression algorithm for compression
  -h, --help
	Show this usage message.
  -v, --verbose
	Display cluster statistics.
",
	   prog_name, prog_name, prog_name);
  exit (rc);
}


#define MIN(a,b) ((a) < (b) ? (a) : (b))
#define MAX(a,b) ((a) > (b) ? (a) : (b))

static inline bool
more_than_one_bit_set (unsigned long x)
{
  return (x & (x - 1));
}


static char const *
basename (char const *full)
{
  char const *p = strrchr (full, '/');
  if (p)
    return p + 1;
  else
    return full;
}


#define strcmpi(x,y) strcasecmp (x, y)

#define streq(x, y) (strcmp (x, y) == 0)


static inline unsigned
round_up (unsigned x, unsigned y)
{
  assert (y > 0);
  assert ((y & (y - 1)) == 0);
  return (((x - 1)
	   & (y - 1))
	  + y);
}

static inline unsigned
round_up_div (unsigned long num, unsigned den)
{
  if (num)
    return ((num - 1) / den) + 1;
  else
    return 0;
}

static inline bool
round_up_ge(unsigned a, unsigned b, unsigned n)
{
  unsigned mask = ~(n - 1);

  assert (!more_than_one_bit_set (n));  /* n must be a power of two. */
  assert (n != 0);
  if (a >= b)	/* We save a bit of arithmetic if this is true. */
    return true;
  if (a == 0)	/* This pretest is necessary to avoid underflow in the below. */
    return false;
  return ((a - 1) & mask) >= ((b - 1) & mask);
}

static unsigned
count_bits (unsigned long x)
{
  register unsigned n = 0;
#define ten_ones ((1 << 10) - 1)

  while ((x & ten_ones) == ten_ones)
    {
      n += 10;
      x >>= 10;
    }

  while ((x & 0xf) == 0xf)
    {
      n += 4;
      x >>= 4;
    }
  
  while (x)
    {
      n++;
      x &= (x - 1);
    }
  return n;
#undef ten_ones
}
  

static bool
is_zero (void const * const o, size_t len)
     /* Returns true iff at least len bytes beginning at o are NULs.
	Returns true if len == 0. */
{
  if (len < sizeof (unsigned))
    {
      register char const *p = (char const *) o;

      while (len--)
	if (*p++)
	  return false;
      return true;
    }
  else
    {
      return !(*((unsigned const *) o)
	       || memcmp ((const void *) o,
			  (const void *) ((unsigned const *) o + 1),
			  len - sizeof (unsigned)));
    }
}


#if 0
/* The following is more portable than the above in that it doesn't require
   casting a char* to a unsigned*. */
static bool
is_zero (char const *p, size_t len)
{
  assert (len > 0);
  return !(*p++
	   || (--len
	       && (*p++
		   || (--len
		       && (*p++
			   || (--len
			       && (*p++
				   || memcmp (o, p, --len))))))));
}
#endif


typedef struct file_status {
  /* Command-line-specified. */
  char const   *path;
  char const   *suffix;
  unsigned	fs_blk_size;
  unsigned	clu_nblocks;
  int		compr_meth;
  bool		do_compress;
  bool		verbose;
  bool		do_continue;
#ifdef BENCH
  bool		do_bench;
  int		nloop;
#endif

  /* Other. */
  off_t		ilen;
  unsigned long	iflags;
  bool		restore_flags;
  bool		iflags_valid;
  int		ifd;
  int		ofd;
  struct utimbuf timestamps;
  unsigned	clu_nchars;	/* Equals (fs_blk_size * clu_nblocks). */
  unsigned long	nclusters;	/* = round_up_div (st.st_size, status->clu_nchars). */
  bool		kernel_e2compr;
#ifdef BENCH
  unsigned long	cmsec;
  unsigned long	umsec;
  unsigned long	cbyte;
  unsigned long	ubyte;
  unsigned long	blk[2];
#endif
} file_status;


/* Always returns -1. */
static int 
fatal_error (file_status const * const status, const char *msg, int err)
{
  unsigned long value;

  if (msg != NULL)
    {
      fprintf (stderr, "%s: %s: %s", prog_name, status->path, msg);

      if (err != 0)
	fprintf (stderr, " (%s)", strerror (err));

      fprintf (stderr, "\n");
    }

  if (status->ifd > STDIN_FILENO)
    {
      if (status->restore_flags)
	{
	  assert (status->iflags_valid); /* fixme: haven't checked this. */
	  value = status->iflags;
	  (void) ioctl (status->ifd, EXT2_IOC_SETFLAGS, &value);
	}

      close (status->ifd);
    }

  if (status->ofd != -1)
    close (status->ofd);

  return -1;
}


static int 
open_files (file_status *status)
{
  unsigned long value;
  struct stat st;

  status->restore_flags	= false;
  status->iflags_valid = false;
  status->ifd = -1;
  status->ofd = -1;

#ifdef BENCH
  status->cmsec = 0;
  status->umsec = 0;
  status->cbyte = 0;
  status->ubyte = 0;
  status->blk[0] = 0;
  status->blk[1] = 0;
#endif

  /*
   *	Open input file.
   */

  if ((status->ifd = open (status->path, O_RDONLY, 0)) == -1)
    return fatal_error (status, "can't open file", errno);

  if (ioctl (status->ifd, (int) EXT2_IOC_GETFLAGS, &value) == -1)
    {
      if (status->do_compress)
	return fatal_error (status, "can't read flags", errno);
    }
  else
    {
      status->iflags = value;
      status->iflags_valid = true;
    }

  if (fstat (status->ifd, &st) == -1)
    return fatal_error (status, "can't stat file", errno);

  if (! S_ISREG (st.st_mode))
    return fatal_error (status, "not a regular file", 0);

  if ((st.st_nlink > 1) && !status->do_continue)
    return fatal_error (status, "file has more than one link", 0);

  status->ilen = st.st_size;

  if (status->do_compress)
    {
      status->kernel_e2compr = true;
      /* This may in fact not be true, but if it turns out to be false then we
         just give an error message and exit. */

      status->clu_nchars = status->clu_nblocks * status->fs_blk_size;
      status->nclusters	= round_up_div ((unsigned long) st.st_size, status->clu_nchars);
    }
  else
    {
      /*
       *	If we are decompressing, we attempt to turn off automatic
       *	decompression for input file, and read the cluster size.
       */
      if (status->iflags_valid
	  && !(status->iflags & (EXT2_COMPR_FL | EXT2_COMPRBLK_FL))
	  && !status->do_continue)
	return fatal_error (status, "file not compressed", 0);

      if (ioctl (status->ifd, (int) EXT2_IOC_GETCLUSTERSIZE, &value) == -1)
	status->kernel_e2compr = false;
      else
	{
	  status->kernel_e2compr = true;
	  status->clu_nblocks = value;
	}

      if (status->iflags_valid)
	{
	  value = status->iflags | EXT2_NOCOMPR_FL;
	  if ((ioctl (status->ifd, EXT2_IOC_SETFLAGS, &value) == -1) && status->kernel_e2compr)
	    return fatal_error (status, "can't turn off compression", errno);

	  status->restore_flags = true;
	}
    }

  /*
   *	Set up output file.
   */
  {
    char opath[strlen (status->path) + strlen (status->suffix) + 1];
    struct stat ost;

    sprintf (opath, "%s%s", status->path, status->suffix);

    if ((stat (opath, &ost) != -1) && !status->do_continue)
      return fatal_error (status, "temporary file already exists", 0);

    if ((status->ofd = open (opath, O_RDWR | O_CREAT, st.st_mode & ~S_IFMT)) == -1)
      return fatal_error (status, "can't open output file", errno);

    if (fstat (status->ofd, &ost) == -1)
      return fatal_error (status, "can't stat temporary file", errno);
    
    if (st.st_dev != ost.st_dev)
      return fatal_error (status, "input and temporary files not on same device", 0);

    if (st.st_ino == ost.st_ino)
      return fatal_error (status, "input and temporary files are the same", 0);

    /* The reason we trunc to 0 before setting the size to ilen is to fill it with
       holes. */
    if (ftruncate (status->ofd, 0) == -1)
      return fatal_error (status, "can't truncate output file", errno);

    status->timestamps.actime = st.st_atime;
    status->timestamps.modtime = st.st_mtime;
  }

  /*
   *	Copy original properties.
   */

  if (status->do_compress)
    {
      /*
       *	If we are compressing, turn off automatic compression
       *	for the output file.
       */

      assert (status->iflags_valid);
      value = status->iflags | EXT2_NOCOMPR_FL | EXT2_COMPR_FL;
      if (ioctl (status->ofd, EXT2_IOC_SETFLAGS, &value) == -1)
	return fatal_error (status, "can't turn off compression", errno);

      value = status->clu_nblocks;
      if (ioctl (status->ofd, EXT2_IOC_SETCLUSTERSIZE, &value) == -1)
	return fatal_error (status, "can't set cluster size", errno);

      value = status->compr_meth;
      if (ioctl (status->ofd, EXT2_IOC_SETCOMPRMETHOD, &value) == -1)
	return fatal_error (status, "can't set compression method", errno);
    }
  else 
    {
      if (status->iflags_valid)
	{
	  value = status->iflags & (unsigned long) ~EXT2_COMPR_FL;
	  if (ioctl (status->ofd, EXT2_IOC_SETFLAGS, &value) == -1)
	    return fatal_error (status, "can't restore original flags", errno);
	}
      else
	{
	  value = 0;
	  if (ioctl (status->ofd, EXT2_IOC_SETFLAGS, &value) == 0)
	    (void) fprintf (stderr,
			    "%s: Interesting circumstance: getflags failed but setflags succeeded.  Please report.\n",
			    prog_name);
	}
    }

  if ((fchown (status->ofd, st.st_uid, st.st_gid) == -1) && !status->do_continue)
    return fatal_error (status, "can't restore original owner/group", errno);

  if (ftruncate (status->ofd, (size_t) status->ilen) == -1)
    return fatal_error (status, "can't truncate output file", errno);

  if (status->verbose)
    fprintf (stderr, "File '%s': %lu bytes, %ld cluster%c\n", 
	     status->path,
	     (unsigned long) status->ilen,
	     status->nclusters,
	     status->nclusters > 1 ? 's' : ' ');

  return 0;
}


/*
 *	Everything is done, we delete the backup file and close
 *	the output file.  Original flags are restored.
 *	Timestamps (mtime and atime) from original file are applied.
 */

static int 
close_files (file_status *status)
{
  unsigned long value;

  if (status->do_compress)
    {
      assert (status->iflags_valid);
      value = (status->iflags | EXT2_COMPR_FL) & (unsigned long) ~EXT2_NOCOMPR_FL;

      if (ioctl (status->ofd, EXT2_IOC_SETFLAGS, &value) == -1)
	return fatal_error (status, "can't set flags", errno);
    }
  else if (status->iflags_valid)
    {
      value = status->iflags;
      (void) ioctl (status->ifd, EXT2_IOC_SETFLAGS, &value);
    }

  /* (opath scope) */
  {
    char opath[strlen (status->path) + strlen (status->suffix) + 1];

    sprintf (opath, "%s%s", status->path, status->suffix);

#ifdef BENCH
    if (do_bench) 
      {
	if (ioctl (status->ofd, EXT2_IOC_GETCOMPRRATIO, &(status->blk)) == -1)
	  return fatal_error (status, "can't get ratio", errno);

	cblk += status->blk[0];
	ublk += status->blk[1];

	unlink (opath);
      }

    else
#endif

      if (rename (opath, status->path) == -1)
	return fatal_error (status, "can't rename file", errno);
  }

  if (utime (status->path, &status->timestamps))
    {
      fprintf (stderr,
	       "%s: Couldn't set timestamps on `%s': %s\n",
	       prog_name, status->path, strerror (errno));
    }

#ifdef BENCH
  if (do_bench)
    {
      printf ("%-12s  %7ld  %7ld   %5.1f",
	      status->path, status->ubyte, status->cbyte, 
	      (status->blk[1] != 0 && status->blk[1] != status->blk[0] 
	       ? (double) 100.0 * status->blk[0] / status->blk[1] 
	       : 100.0));
    
      if (status->cmsec > 0)
	printf ("  %5.2f MB/s",
		(double) status->ubyte * status->nloop / (1000.0 * status->cmsec));
      else 
	printf ("   ?.?? MB/s");

      if (status->umsec > 0)
	printf (" %5.2f MB/s",
		(double) status->ubyte * status->nloop / (1000.0 * status->umsec));
      else 
	printf ("  ?.?? MB/s");

      if (status->umsec < 500 || status->cmsec < 500)
	printf ("  *\n"); 
      else 
	printf ("\n");
    }

  cmsec += status->cmsec;
  umsec += status->umsec;
  cbyte += status->cbyte;
  ubyte += status->ubyte;
#endif

  close (status->ifd);
  close (status->ofd);

  return 0;
}


static inline int
mflcv_verify_compression (size_t const ulen,
			  size_t const clen,
			  file_status * const status,
			  unsigned char const * const ubuf,
			  unsigned char * const cbuf)
{
  size_t vlen;
  static unsigned char *vbuf = NULL;
#ifdef BENCH
  struct rusage old, new;
#endif

  /* effic: We can probably share ubuf or cbuf. */
  if ((vbuf = realloc (vbuf, status->clu_nchars)) == NULL)
    return fatal_error (status, "cannot allocate vbuf", 0);

#ifdef BENCH
  getrusage (RUSAGE_SELF, &old);
	  
  for (i = 0; i < status->nloop; i++)
    {
#endif
      if (algo[status->compr_meth].init (heap_base) > HEAP_SIZE)
	return fatal_error (status, "heap not big enough !!!", 0);

      vlen = algo[status->compr_meth].uncompress (cbuf + sizeof (struct ext2_cluster_head),
						  vbuf,
						  clen,
						  ulen,
						  algo[status->compr_meth].xarg);

#ifdef BENCH
    }

  getrusage (RUSAGE_SELF, &new);
  status->umsec += ((new.ru_utime.tv_sec - old.ru_utime.tv_sec) * 1000
		    + new.ru_utime.tv_usec - old.ru_utime.tv_usec);
#endif
  /* tbi: Can ulen equal zero at this point? */
  if ((vlen <= 0) || (vlen != ulen) || (memcmp (ubuf, vbuf, ulen) != 0))
    return fatal_error (status, "verification failed", 0);

  return 0;
}

static unsigned long
mflch_create_head (unsigned const ulen,
		   unsigned const clen,
		   unsigned long const ubitmap,
		   unsigned char const * const ubuf,
		   unsigned char * const cbuf,
		   int compr_meth)
     /* Returns adler32 checksum. */
{
  struct ext2_cluster_head *head  = (struct ext2_cluster_head *) cbuf;
  head->magic	    = EXT2_COMPRESS_MAGIC;
  head->len	    = ulen;
  head->compr_len   = clen;
  assert ((compr_meth >= 0) && (compr_meth <= 24));
  head->method	    = compr_meth;
  head->reserved_0  = 0;
  head->bitmap	    = ubitmap;
  return head->reserved_2 = ext2_adler32 (1, ubuf, ulen);
}


static inline int
mflc_compress_cluster (unsigned long const cluster,
		       struct file_status * const status,
		       unsigned const ulen,
		       int * const oleft_p,
		       unsigned char const ** const optr_p,
		       unsigned long const ubitmap,
		       unsigned long * const obitmap,
		       unsigned char * const ibuf,
		       unsigned char * const obuf)
{
  size_t clen;
#ifdef BENCH
  int i;
  struct rusage old;
  struct rusage new;
#endif

  if (ulen <= status->fs_blk_size)
    clen = 0;   /* Won't compress to a smaller number of blocks. */
  else
    {
#ifdef BENCH
      getrusage (RUSAGE_SELF, &old);

      for (i = 0; i < status->nloop; i++)
	{
#endif
	  if (algo[status->compr_meth].init (heap_base) > HEAP_SIZE)
	    return fatal_error (status, "heap not big enough !!!", 0);

	  clen = algo[status->compr_meth].compress (ibuf, 
						    obuf + sizeof (struct ext2_cluster_head), 
						    ulen, 
						    (((ulen - 1)
						      & ~(status->fs_blk_size - 1))
						     - sizeof (struct ext2_cluster_head)),
						    algo[status->compr_meth].xarg);

#ifdef BENCH
	}
	  
      getrusage (RUSAGE_SELF, &new);
	  
      status->cmsec += ((new.ru_utime.tv_sec - old.ru_utime.tv_sec) * 1000
			+ new.ru_utime.tv_usec - old.ru_utime.tv_usec);
#endif
    }

#ifdef BENCH
  status->ubyte += ulen;
  status->cbyte += clen;
#endif

  if (clen == 0)
    {   	/* Couldn't get it any smaller. */
      if (ioctl (status->ofd, EXT2_IOC_CLRCLUSTERBIT, &cluster) == -1)
	return fatal_error (status, "can't clear cluster bit", errno);

      /* tbi: What happens if we try to clear bit n for n > 8208?  I
	 believe that correct behaviour for 8208 <= n < 2**31 would be
	 to succeed for EXT2_IOC_CLRCLUSTERBIT but fail for
	 EXT2_IOC_SETCLUSTERBIT.  I don't know what actually
	 happens. */

      *optr_p = ibuf;
      *oleft_p = ulen;
      *obitmap = ubitmap;

      if (status->verbose)
	{
	  int nblks = round_up_div (ulen, status->fs_blk_size);

	  fprintf (stderr, " cluster #%3ld left uncompressed.  ubitmap=0x%08lx, blocks=%2d\n",
		   cluster, ubitmap, nblks);
	}
    }
  else
    {
      unsigned long adler32;

      if (ioctl (status->ofd, EXT2_IOC_SETCLUSTERBIT, &cluster) == -1)
	{
	  return fatal_error (status, "can't set cluster bit", errno);

	  /* TODO: More graceful behaviour (considering the limited number
	     of cluster bits in current kernel implementation) would be to
	     print a warning and copy the rest of the file in its
	     uncompressed form. */
	}

      adler32 = mflch_create_head (ulen, clen, ubitmap, ibuf, obuf, status->compr_meth);
	  
      if (!status->do_continue
#ifdef BENCH
	  || status->do_bench
#endif
	  )
	if (mflcv_verify_compression (ulen, clen, status, ibuf, obuf))
	  return -1;

      /* I believe that output block positions must be a subset of the
         positions of the uncompressed blocks. */
      {
	int nblks = round_up_div (clen + sizeof (struct ext2_cluster_head), 
				  status->fs_blk_size);
	unsigned long mask = (1 << nblks) - 1;

	if ((ubitmap & mask) == mask)
	  *obitmap = mask;
	else
	  {
	    unsigned long killbits = ubitmap;

	    while (nblks--)
	      killbits &= (killbits - 1);
	    assert ((killbits & ubitmap) == killbits);
	    *obitmap = ubitmap ^ killbits;
	    assert ((ubitmap & *obitmap) == *obitmap);
	  }
      }
      *optr_p = obuf;
      *oleft_p = clen + sizeof (struct ext2_cluster_head);

      if (status->verbose)
	fprintf (stderr, " cluster #%3ld, ubitmap=0x%08lx, blocks=%2d/%2d, adler32=0x%08lx\n",
		 cluster, ubitmap, 
		 round_up_div (clen + sizeof (struct ext2_cluster_head), 
			       status->fs_blk_size),
		 round_up_div (ulen, status->fs_blk_size),
		 adler32);
    }

  return 0;
}


static int
write_cluster (
	       file_status const * const  status,
	       unsigned  		   oleft,    /* N. chars to write;
						        length of significant
						        chars from optr.  Except
						        for the last cluster,
						        oleft = (n. bits set in
						        obitmap) *
						        status->fs_blk_size. */
	       char const *		   optr,     /* Pointer to cluster chars. */
	       unsigned long const	   obitmap,  /* How to distribute the chars. */
	       off_t *  		   offsetp)  /* Where in the file to
						        start writing.  We add
						        status->clu_nchars to
						        offsetp (even at end of
						        file). */
{
  int block;

  /*
   *	Write back the cluster.
   */

  /* Make sure clu_nblocks, fs_blk_size and clu_nchars have valid values. */
  assert ((status->clu_nblocks >= 4) && (status->clu_nblocks <= 32));
  assert (status->fs_blk_size >= EXT2_MIN_BLOCK_SIZE);
  assert (status->fs_blk_size <= EXT2_MAX_BLOCK_SIZE);
  assert (status->clu_nchars == status->clu_nblocks * status->fs_blk_size);

  for (block = 0; block < status->clu_nblocks; block++)
    {
      if (obitmap & (1 << block))
	{
	  unsigned write_len = MIN (oleft, status->fs_blk_size);

	  assert (oleft > 0);

	  if (lseek (status->ofd, *offsetp, SEEK_SET) != *offsetp)
	    return fatal_error (status, "can't seek output file", errno);

	  if (write (status->ofd, optr, write_len) != write_len)
	    return fatal_error (status, "can't write to output file", errno);

	  oleft -= write_len;
	  optr  += write_len;
	}
	
      *offsetp += status->fs_blk_size;
    }
  return 0;
}	


enum warn_cmd {
  warn_rsrvd0, warn_long_ulen, warn_clen, warn_bitcount, warn_misaligned_ulen,
  warn_uncmp_head, warn_compression_error, warn_checksum_error, 
  something_very_wrong, warn_not_hole,
  n_warnings,	/* This is not a warning but a count. */
  begin_stream, end_stream
};

static char const *
warn_str (enum warn_cmd cmd)
{
  switch (cmd)
    {
    case warn_rsrvd0:
      return "Non-zero reserved_0 field in head.  This is expected in about 1 in 100 000 uncompressed clusters.";
    case warn_long_ulen:
      return "Uncompressed cluster len exceeds 32KB.";
    case warn_clen:
      return "Compressed data would take up at least as many blocks as uncompressed.";
    case warn_bitcount:
      return "Bitmap seems to conflict with uncompressed length field.";
    case warn_misaligned_ulen:
      return "Length not a multiple of determined minimum block size.";
    case warn_uncmp_head:
      return "Cluster bit provided by kernel indicates that this cluster is not considered compressed.";
    case warn_compression_error:
      return "The head appears valid, but the data couldn't be uncompressed.";
    case warn_checksum_error:
      return "Checksum error.";
    case something_very_wrong:
      return "Cluster decompressed OK, but the information conflicts with a previous cluster.  An error has probably been made.\n";
    case warn_not_hole:
      return "Non-null data read where a hole was expected.";
    case n_warnings:
    case begin_stream:
    case end_stream:
      assert (0 && "unexpected warning");
      abort();
    }
  assert (0 && "can't reach here");
  return NULL;
}

static void
warn (enum warn_cmd cmd, char const *id)
{
  static int warn_count[n_warnings];

  switch (cmd)
    {
    case begin_stream:
      (void) memset (warn_count, 0, sizeof (*warn_count) * n_warnings);
      return;
    case end_stream:
      {
	int i;
	bool head_printed = false;

	for (i = 0; i < n_warnings; i++)
	  if (warn_count[i] != 0)
	    {
	      if (!head_printed)
		{
		  (void) fprintf (stderr,
				  "%s: The following probable error(s) occurred while decompressing %s:\n",
				  prog_name, id);
		  head_printed = true;
		}
	      (void) fprintf (stderr, "  %s\n", warn_str (i));
	      if (warn_count[i] > 1)
		(void) fprintf (stderr, "  (Last message occurs %d times.)\n", warn_count[i]);
	    }
      }
      return;
    default:
      warn_count[cmd]++;
      return;
    }
  assert (0 && "Can't reach here.");
}


static int
decompress_stream (file_status * const status,
		   const char * const ichars,   /* Pointer to mmap'ed file. */
		   char * const ibuf,
		   char * const obuf)
{
  unsigned clu_min_nchars, clu_max_nchars;
  unsigned fsblk_min_nchars, fsblk_max_nchars;
  unsigned long i_ix;

  /* todo: Get the ioctl that returns fs_blk_size in e2compr kernels. */
  fsblk_min_nchars = EXT2_MIN_BLOCK_SIZE;
  fsblk_max_nchars = EXT2_MAX_BLOCK_SIZE;

  /* Note: if we were compressing, we'd have the additional restriction
     that the cluster size be at least as large as the page size, but this
     isn't relevant to decompressing in user space. */
  if (status->kernel_e2compr)
    {
      clu_min_nchars = status->clu_nblocks * fsblk_min_nchars;
      clu_max_nchars = MIN (status->clu_nblocks * fsblk_max_nchars, 32768);
      assert (clu_min_nchars >= 4096);
      assert (clu_max_nchars <= 32768);
      assert (clu_min_nchars <= clu_max_nchars);
    }
  else
    {
      /* The minimum is max(4 * minimum ext2fs block size, smallest possible page size). */
      clu_min_nchars = 4096;
      clu_max_nchars = 32768;
    }

  /* N.B. In the following I've assumed that it's possible for a
     cluster that is compressed to be read as uncompressed.  This
     possibility is the result of a bug in the e2compr kernel code,
     whereby the NOCOMPR flag is not honoured for reads.

     At the time of writing, there are 3 kinds of kernel (w.r.t. e2compr)
     known to exist:

     i) No e2compr support.

     ii) Has e2compr support, but the NOCOMPR flag never honoured for
     reading.  This is the state of the publically-released code.

     iii) Has e2compr support, and the NOCOMPR flag is partially
     supported for reading.  This is the state of my own kernel at
     present.  This incomplete state is the worst of the three ---I
     believe that it's possible (though fortunately very rare) for a
     multi-page cluster to be read part compressed and part
     uncompressed :-( --- so it won't be around for long.  When it's
     gone, I can focus the code on the other possibilities.

     The expected fourth kind is where e2compr is supported and reads
     with NOCOMPR are correctly implemented. */

  warn (begin_stream, status->path);

  /* Main loop. */
  for (i_ix = 0; i_ix < status->ilen;)
    {
      const char *iptr = ichars + i_ix;
      struct ext2_cluster_head const *head = ((struct ext2_cluster_head const *) 
					      iptr);
					    
      /* Consider the cluster to be compressed until we find otherwise. */

      if (head->magic != EXT2_COMPRESS_MAGIC)
	goto consider_uncompressed;

      if (head->reserved_0 != 0)
	{
	  warn (warn_rsrvd0, status->path);
	  goto consider_uncompressed;
	}

      assert (sizeof (head->len) * BITSPERBYTE == 16);
      if ((unsigned) head->len > 32768)
	{
	  warn (warn_long_ulen, status->path);
	  goto consider_uncompressed;
	}

      if (round_up_ge (head->compr_len, head->len, fsblk_min_nchars))
	{
	  /* Neither the kernel nor e2compress(1) allow compressed data to take
	     up as much space as the uncompressed data. */
	  warn (warn_clen, status->path);
	  goto consider_uncompressed;
	}
      
      /* Except for the last cluster, the number of bits set in bitmap should equal
	 (len * fs_blksize). */
      if (head->bitmap == 0)
	{
	  warn (warn_bitcount, status->path);
	  goto consider_uncompressed;
	}
      if (status->ilen - i_ix < clu_min_nchars)
	{
	  unsigned nbits = count_bits (head->bitmap);

	  /* fsblk_nchars >= floor (len / nbits)
	     fsblk_nchars < ceil (len / (nbits - 1))
	     See below for reasoning. */
	  if (fsblk_max_nchars < head->len / nbits)
	    {
	      warn (warn_bitcount, status->path);
	      goto consider_uncompressed;
	    }
	  if (fsblk_min_nchars > (head->len - 1) / (nbits - 1) + 1)
	    {
	      warn (warn_bitcount, status->path);
	      goto consider_uncompressed;
	    }
	}
      else
	{
	  unsigned tmp;

	  if (head->len & (fsblk_min_nchars - 1))
	    {
	      warn (warn_misaligned_ulen, status->path);
	      goto consider_uncompressed;
	    }

	  tmp = head->len / count_bits (head->bitmap);
	  if ((tmp & (tmp - 1))
	      || (tmp < fsblk_min_nchars)
	      || (tmp > fsblk_max_nchars))
	    {
	      warn (warn_bitcount, status->path);
	      goto consider_uncompressed;
	    }
	}
      
      if (status->kernel_e2compr)
	{
	  unsigned long value = i_ix / status->clu_nchars;
    
	  assert (i_ix >= 0);
	  assert (i_ix < status->ilen);

	  if (ioctl (status->ifd, (int) EXT2_IOC_GETCLUSTERBIT, &value) == -1)
	    return fatal_error (status, "can't read cluster bit", errno);
	  else if (!value)
	    {
	      warn (warn_uncmp_head, status->path);
	      goto consider_uncompressed;
	    }
	}

      if (algo[head->method].init (heap_base) > HEAP_SIZE)
	return fatal_error (status, "heap not big enough !!!", 0);

      /* (ulen scope) */
      {
	unsigned ulen;

	ulen = algo[head->method].uncompress (ibuf + sizeof (struct ext2_cluster_head),
					      obuf,
					      head->compr_len,
					      head->len,
					      algo[status->compr_meth].xarg);

	if (ulen != head->len)
	  {
	    warn (warn_compression_error, status->path);
	    goto consider_uncompressed;
	  }
      }
      
      if ((head->reserved_2 != 0)
	  && (ext2_adler32 (1, obuf, head->len) != head->reserved_2))
	{
	  warn (warn_checksum_error, status->path);
	  goto consider_uncompressed;
	}
  
      /* Only now do we update clu_min_nchars and clu_max_nchars. */
      {
	/* clu_max_nchars */
	{
	  unsigned long suggested_clu_max_nchars = ((i_ix & (i_ix - 1)) 
						    ^ i_ix);
	  /* The above var has only the lowest bit of i_ix set. */
	  if ((i_ix != 0) && (suggested_clu_max_nchars < clu_max_nchars))
	    clu_max_nchars = suggested_clu_max_nchars;
	}
  
	/* clu_min_nchars */
	if (head->len > clu_min_nchars)
	  {
	    register unsigned this_len = head->len;
        
	    assert (clu_min_nchars >= 4096);
	    if (this_len > 16384)
	      clu_min_nchars = 32768;
	    else if (this_len > 8192)
	      clu_min_nchars = 16384;
	    else
	      clu_min_nchars = 8192;
	  }
      }

      /* effic: I believe this impossible; prove it. */
      if (clu_min_nchars > clu_max_nchars)
	{
	  (void) fprintf (stderr, "\
%s: Either the file `%s' contains an extremely improbable 
bit pattern, or this program contains a bug.  Please send a bug report, 
along with a copy of the input file.\n", prog_name, status->path);
	  return -1;
	}
    
      /* Update fsblk_min_nchars and fsblk_max_nchars. */
      {
	int nbits = count_bits (head->bitmap);

	/* Except for the last cluster, uncompressed len should equal
           the number of bits set in bitmap multiplied by the filesystem
           block size.  In the last cluster, this product equals the
           rounded up uncompressed len. */
	assert (head->bitmap != 0);
	assert ((fsblk_min_nchars & (fsblk_min_nchars - 1)) == 0);
	assert ((fsblk_max_nchars & (fsblk_max_nchars - 1)) == 0);
	assert (fsblk_min_nchars >= EXT2_MIN_BLOCK_SIZE);
	assert (fsblk_max_nchars <= EXT2_MAX_BLOCK_SIZE);
	assert (fsblk_max_nchars >= fsblk_min_nchars);
	if (status->ilen - i_ix < clu_min_nchars)
	  {
	    /* We can't find an exact value, but we can at least narrow down the
               upper limit. */
	    assert (head->compr_len < head->len);
	    assert (!round_up_ge (head->compr_len, head->len, fsblk_min_nchars));

	    /* len <= nbits * fsblk_nchars              (1)
	       len > (nbits - 1) * fsblk_nchars         (2)
	       nbits >= 1                               (3)
	       fsblk_nchars & (fsblk_nchars - 1) == 0   (4)
	       (1) /\ (3) ==>  fsblk_nchars >= len / nbits       (5)
	       (2) /\ (3) ==>  fsblk_nchars < len / (nbits - 1)  (6)
	       (5) ==>  fsblk_nchars >= floor (len / nbits)
	       (6) ==>  fsblk_nchars < ceil (len / (nbits - 1)) */

	    while (round_up_ge (head->compr_len, head->len, fsblk_max_nchars))
	      fsblk_max_nchars /= 2;

	    if (nbits > 1)
	      {
		int tmp = (head->len - 1) / (nbits - 1);
		/* tbi: Document the optimisation based on (n >= expr + 1) <==>
                   (n > expr) for integers.  Also show that ceil (num / den)
                   == 1 + floor ((num - 1) / den) for all positive integer
                   (num, den). */

		if (fsblk_min_nchars > tmp)
		  {
		    warn (something_very_wrong, status->path);
		    goto consider_uncompressed;
		  }

		while (fsblk_max_nchars > tmp)
		  fsblk_max_nchars /= 2;
	      }

	    /* (tmp scope) */
	    {
	      unsigned tmp = head->len / nbits;

	      if (fsblk_max_nchars < tmp)
		{
		  warn (something_very_wrong, status->path);
		  goto consider_uncompressed;
		}

	      while (fsblk_min_nchars < tmp)
		fsblk_min_nchars *= 2;
	    }
	  }
	else
	  {
	    unsigned tmp;

	    tmp = head->len / nbits;
	    if ((tmp & (tmp - 1))
		|| (tmp < fsblk_min_nchars)
		|| (tmp > fsblk_max_nchars))
	      {
		warn (warn_bitcount, status->path);
		goto consider_uncompressed;
	      }

	    if (round_up_ge (head->compr_len, head->len, tmp))
	      {
		warn (something_very_wrong, status->path);
		goto consider_uncompressed;
	      }

	    /* The minimum cluster size is 4 filesystem blocks. */
	    if (4 * tmp > clu_max_nchars)
	      {
		warn (something_very_wrong, status->path);
		goto consider_uncompressed;
	      }

	    fsblk_min_nchars = fsblk_max_nchars = tmp;
	  }      
      }

      /* Check that remaining blocks in cluster are zeroed (i.e. holes). */
      {
	unsigned occupied_nchars = round_up ((sizeof (struct ext2_cluster_head)
					      + head->compr_len),
					     fsblk_max_nchars);
	char const *zero_start = (iptr + occupied_nchars);
	unsigned zero_len = MIN (clu_min_nchars - occupied_nchars,
				 status->ilen - i_ix);

	assert (clu_min_nchars >= occupied_nchars + fsblk_max_nchars);

	/* Hmm, it's still possible for zero_len to be less than 4 if the file
           size is n bytes more than a multiple of the filesystem blocksize,
           where 1 <= n < 4. */
	assert (zero_len >= 1);
	if ((*zero_start != '\0')
	    || ((zero_len > 1) && (zero_start[1] != '\0'))
	    || ((zero_len > 2) && (zero_start[2] != '\0'))
	    || ((zero_len > 3) && (zero_start[3] != '\0'))
	    || ((zero_len > 3) && memcmp (zero_start, zero_start + 4, zero_len - 4)))
	  warn (warn_not_hole, status->path);
	goto consider_uncompressed;
	/* I'm not sure about this, but the idea is that we don't lose the data
	   kept in what's supposed to be a hole.  This should almost certainly
	   be reported, preferably with offset. */
      }
  
      if (status->verbose)
	{
	  fprintf (stderr, " cluster offset %5ld, obitmap=0x%08lx, blocks=%2u/%2u, adler=0x%08lx\n",
		   (long) i_ix,
		   (unsigned long) head->bitmap, 
		   (unsigned) round_up_div ((head->compr_len 
					     + sizeof (struct ext2_cluster_head)), 
					    fsblk_min_nchars),
		   (unsigned) round_up_div (head->len, fsblk_min_nchars),
		   (unsigned long) head->reserved_2);

	  /* tbi: Is fsblk_min_nchars guaranteed to equal fsblk_max_nchars?  If
             not, should we give some indication of the unsurety? */
	}

      status->fs_blk_size = fsblk_min_nchars;
      status->clu_nchars = clu_min_nchars;
      status->clu_nblocks = clu_min_nchars / fsblk_min_nchars;
      if (write_cluster (status, head->len, obuf, head->bitmap, &i_ix))
	return -1;
  
      goto next_block;

    consider_uncompressed:
      {
	/* Cluster is already stored in uncompressed form: copy straight to output. */
	size_t nchars;
	unsigned long bitmap = 0;
	unsigned long cluster_offset = i_ix;

	nchars = MIN (status->ilen - i_ix, clu_min_nchars);

	/* Write it out, handling holes. */
	{
	  unsigned long bit = 1;
	  unsigned oleft = nchars;

	  assert (nchars / 32 <= fsblk_min_nchars);
	  while (oleft)
	    {
	      unsigned write_len = MIN (oleft, fsblk_min_nchars);

	      assert (ichars + i_ix == iptr);
	      if (is_zero (iptr, write_len))
		;
	      else
		{
		  if (lseek (status->ofd, (off_t) i_ix, SEEK_SET) != i_ix)
		    return fatal_error (status, "can't seek output file", errno);

		  if (write (status->ofd, iptr, write_len) != write_len)
		    return fatal_error (status, "can't write to output file", errno);

		  bitmap |= bit;
		}
	      oleft -= write_len;
	      iptr += write_len;
	      i_ix += write_len;
	      bit <<= 1;
	    }
	}

	if (status->verbose)
	  {
	    if (fsblk_min_nchars == fsblk_max_nchars)
	      fprintf (stderr, " cluster at offset %lu wasn't compressed.  bitmap=%08lx.\n",
		       cluster_offset, bitmap);
	    else
	      fprintf (stderr, " cluster at offset %lu wasn't compressed.  bitmap=%08lx if fs blksize = %d.\n",
		       cluster_offset, bitmap, fsblk_min_nchars);
	  }
      }

    next_block:
      ;
    } /* for */
  warn (end_stream, status->path);
  return 0;
}


static inline int
mfl_process_cluster (unsigned long const cluster,
		     struct file_status * const status,
		     unsigned char * const ibuf,
		     unsigned char * const obuf,
		     off_t *ileftp,
		     off_t *offsetp)
{
  unsigned oleft;
  unsigned char const *optr;
  unsigned char *iptr = ibuf;
  unsigned long ibitmap = 0;
  unsigned long obitmap;
  unsigned ilen = 0;

  /*
   *	Read the cluster, creating ibitmap as we go.
   */
  {
    int block;
    
    for (block = 0; (block < status->clu_nblocks) && (*ileftp > 0); block++)
      {
	unsigned chars = (*ileftp <= status->fs_blk_size
			  ? *ileftp
			  : status->fs_blk_size);

	iptr[0] = iptr[1] = iptr[2] = iptr[3] = 0;
	if (read (status->ifd, iptr, chars) != chars)
	  return fatal_error (status, "can't read from input file", errno);

	if ((iptr[0] != 0)
	    || (iptr[1] != 0)
	    || (iptr[2] != 0)
	    || (iptr[3] != 0)
	    || ((chars > 4)
		&& (memcmp (iptr, iptr + 4, chars - 4) != 0)))
	  {
	    ibitmap |= 1 << block;
	    iptr += chars;
	    ilen += chars;
	  }

	*ileftp -= chars;
      }
  }

  assert (status->do_compress);
  if (mflc_compress_cluster (cluster, status, ilen, &oleft, &optr, 
			     ibitmap, &obitmap, ibuf, obuf))
    return -1;

  if (write_cluster (status, oleft, optr, obitmap, offsetp))
    return -1;
    
  return 0;
}


static int 
mf_process_file (file_status *status)
{
  unsigned long cluster;
  static unsigned char *ibuf = NULL;
  static unsigned char *obuf = NULL;

  if (open_files (status) != 0)
    return -1;

  if (heap_base == NULL && (heap_base = malloc (HEAP_SIZE)) == NULL)
    return fatal_error (status, "can't allocate heap", 0);

  /* Allocate working buffers. */
  {
    size_t bufsize;

    if (status->do_compress)
      bufsize = status->clu_nchars;
    else
      bufsize = 32768;  /* Maximum cluster size. */

    if ((ibuf = realloc (ibuf, bufsize)) == NULL)
      return fatal_error (status, "can't allocate buffers", 0);

    if ((obuf = realloc (obuf, bufsize)) == NULL)
      return fatal_error (status, "can't allocate buffers", 0);
  }

  if (status->do_compress)
    {
      off_t offset = 0;
      off_t ileft = status->ilen;

      for (cluster = 0; cluster < status->nclusters; cluster++)
	if (mfl_process_cluster (cluster, status, ibuf, obuf, &ileft, &offset))
	  return -1;
    }
  else
    {
      int rc;
      void *p = mmap (NULL, (size_t) status->ilen, PROT_READ, MAP_PRIVATE, status->ifd, (off_t) 0);

      if (p == (void *) (-1))
	return fatal_error (status, "can't mmap input file", errno);

      rc = decompress_stream (status, p, ibuf, obuf);

#ifdef NDEBUG
      munmap (p, status->ilen);
#else
      assert (munmap (p, (size_t) status->ilen) == 0);
#endif
      if (rc)
	return rc;
    }
  close_files (status);
  return 0; 
}


int 
main (int argc, char *argv[])
{
  int c, i;
  int n_files = 0;

  /* todo: Create an ioctl that returns these at runtime. */
  unsigned clu_nblocks = EXT2_DEFAULT_CLUSTER_SIZE;
  int compr_meth = EXT2_DEFAULT_COMPR_METHOD;
  bool do_compress;
  bool verbose = false;
  bool do_continue = false;
  int fs_blk_size = 1024;
  const char *suffix = "~z~";
#ifdef BENCH
  bool do_bench = false;
  int nloop = 10;
#endif

  file_status *status_array = NULL;
  int n_status_alloc = 0;


  prog_name = basename (*argv);

  do_compress = !((strcmpi (prog_name, "e2uncompress") == 0)
		  || (strcmpi (prog_name, "e2decompress") == 0));

  for (;;)
    {
      char *tmp;
      struct option longopts[] = {
	{"method", required_argument, NULL, (int) 'm'},
	{"algorithm", required_argument, NULL, (int) 'm'},
#ifdef BENCH
	{"benchmark", no_argument, NULL, (int) 'B'},
	{"loops", required_argument, NULL, (int) 'N'},
#endif
	{"fs-blocksize", required_argument, NULL, (int) 'F'},
	{"suffix", required_argument, NULL, (int) 'S'},
	{"cluster-blocks", required_argument, NULL, (int) 'b'},
	{"decompress", no_argument, NULL, (int) 'd'},
	{"uncompress", no_argument, NULL, (int) 'd'},
	{"force", optional_argument, NULL, (int) 'f'},
	{"help", no_argument, NULL, (int) 'h'},
	{"verbose", no_argument, NULL, (int) 'v'},
	{"version", no_argument, NULL, 0}
      };
      int option_ix = 0;

      c = getopt_long (argc, argv, "-m:A:BtF:N:S:b:df::hv", longopts, &option_ix);
      if (c == EOF)
	{
	  assert (optind == argc);
	  break;
	}

      switch (c) 
	{
	case 0:
	  if (streq (longopts[option_ix].name, "version"))
	    return (fprintf (stderr, "%s\n", version_string) < 0);
	  if (streq (longopts[option_ix].name, "compress"))
	    {
	      do_compress = true;
	      break;
	    }
	  (void) fprintf (stderr, "%s: bug: unrecognised option `%s'.  Please report.\n",
			  prog_name, longopts[option_ix].name);
	  abort ();

	case 'd': do_compress = false; break;
	case 'f': do_continue = 1; break;
	case 'v': verbose     = 1; break;
#ifdef BENCH
	case 'B': do_bench    = 1; break;
	case 'N':
	  nloop = strtol (optarg, &tmp, 0);
	  if (*tmp)
	    usage (1);
	  break;
#endif
	case 'F':
	  fs_blk_size = strtol (optarg, &tmp, 0);
	  if (*tmp)
	    usage (1);
	  if ((fs_blk_size & (fs_blk_size - 1))
	      || (fs_blk_size < EXT2_MIN_BLOCK_SIZE)
	      || (fs_blk_size > EXT2_MAX_BLOCK_SIZE))
	    {
	      fprintf (stderr,
		       "%s: invalid fs blocksize `%s'.\n"
		       "  Ought to be a power of two in the range ["
		       __STRING(EXT2_MIN_BLOCK_SIZE)
		       ", "
		       __STRING(EXT2_MAX_BLOCK_SIZE)
		       "].\n",
		       prog_name, argv[optind]);
	      exit (1);
	    }
	  break;

	case 'b':
	  clu_nblocks = strtol (optarg, &tmp, 0);
	  if (*tmp || (clu_nblocks !=  4 && clu_nblocks !=  8 &&
		       clu_nblocks != 16 && clu_nblocks != 32))
	    {
	      fprintf (stderr, "%s: invalid cluster size: %s\n",
		       prog_name, argv[optind]);
	      exit (1);
	    }
	  break;
	
	case 'S':
	  suffix = optarg;
	  break;

	case 'm':
	case 'A':
	  {
	    int x;
	    
	    for (x = 0; name2id[x].name != NULL; x++)
	      if (strcasecmp (optarg, name2id[x].name) == 0)
		break;
	    
	    if (name2id[x].name == NULL)
	      {
		fprintf (stderr, "%s: unknown algorithm: %s\n", prog_name, optarg);
		exit (1);
	      }

	    compr_meth = name2id[x].id;
	  }
	break;

	case 'h':
	  usage (0);

	case 1:
	  /* Expand status_array if necessary. */
	  assert (n_files <= n_status_alloc);
	  if (n_files == n_status_alloc)
	    {
	      if (n_status_alloc)
		n_status_alloc *= 4;
	      else
		n_status_alloc = 1;

	      status_array = realloc (status_array, sizeof (file_status) * n_status_alloc);
	      if (status_array == NULL)
		{
		  (void) fprintf (stderr, "%s: failed to realloc %lu bytes for status array.\n",
				  prog_name, (unsigned long) n_status_alloc * sizeof (file_status));
		  return 1;
		}
	    }

	  /* Copy current options to status. */
	  status_array[n_files].path = optarg;
	  status_array[n_files].suffix = suffix;
	  status_array[n_files].fs_blk_size = fs_blk_size;
	  status_array[n_files].clu_nblocks = clu_nblocks;
	  status_array[n_files].compr_meth = compr_meth;
	  status_array[n_files].do_compress = do_compress;
	  status_array[n_files].verbose = verbose;
	  status_array[n_files].do_continue = do_continue;
#ifdef BENCH
	  if (do_bench)	
	    {
	      if (!do_compress)
		{
		  (void) fprintf (stderr, 
				  "%s: benchmarking not supported with decompression.\n", 
				  prog_name);
		  (void) fprintf (stderr, "No files processed.\n");
		  return 1;
		}
	      status_array[n_files].do_bench = true;
	      status_array[n_files].nloop = nloop;
	    }
	  else
	    {
	      status_array[n_files].do_bench = false;
	      status_array[n_files].nloop = 1;
	    }
#endif

	  n_files++;
	  break;

	default:
	  usage (1);
	}
    }

  if (n_files == 0)
    usage (1);

  /* TODO: We could check if some options are unused. */

#ifdef BENCH
  if (do_bench)  /* fixme: should be: `If at least one file was processed with benchmarking info'. */
    {
      printf ("--------------------------------------\n");
# if 0
      /* tbi.  We used to print the options here, but that's a bit difficult if the options can vary
	 from file to file. */
      for (i = 0; i < optind; i++)
	printf ("%s ", argv[i]);
      printf ("\n\n");
# endif
      printf ("File name      Length            Ratio\n");
      printf ("---------      ------            -----\n");
    }
#endif

  for (i = 0; i < n_files; i++)
    mf_process_file (&status_array[i]);

#ifdef BENCH
  if (do_bench) /* fixme: ditto, should only apply to benched files. */
    {
      printf ("---------      ------            -----\n");
      printf ("%-7s [%2d]  %7ld  %7ld   %5.1f",
	      algo[compr_meth].name, cluster_nblocks, ubyte, cbyte, 
	      ((ublk != 0) && (cblk != ublk)
	       ? (double) 100.0 * cblk / ublk
	       : 100.0));

      /* fixme: nloop in the below. */
      if (cmsec > 0)
	printf ("  %5.2f MB/s", (double) ubyte * nloop / (1000.0 * cmsec));
      else
	printf ("  ??.?? MB/s");
	      
      if (umsec > 0)
	printf (" %5.2f MB/s\n", (double) ubyte * nloop / (1000.0 * umsec));
      else
	printf ("  ??.?? MB/s\n");
    }
#endif

  return 0;
}
