/*
 * This is an alternative chattr to the one you get by applying the 
 * e2fsprogs patch.  It is pretty much the same feature-wise (except
 * no verbose flag and ignores files that aren't either regular files
 * or directories), but it's more likely to be free from last-minute  bugs.
 */

/*
 * chattr.c		- Change file attributes on an ext2 file system
 *
 * Copyright (C) 1993, 1994  Remy Card <card@masi.ibp.fr>
 *                           Laboratoire MASI, Institut Blaise Pascal
 *                           Universite Pierre et Marie Curie (Paris VI)
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/10/30	- Creation
 * 93/11/13	- Replace stat() calls by lstat() to avoid loops
 * 94/02/27	- Integrated in Ted's distribution
 *
 * 95/02/17	- Completly rewriten to enable control over compression
 *		  parameters. Make sure we don't close the file after setting
 *		  the compress flag, and before setting the cluster size and
 *		  compression algorithm. Uses less space in the stack for the
 *		  path name when recursive.
 *
 *		  	Antoine de Maricourt.
 *
 * 96/10/xx	- Small changes, correct `-R' bug.
 */

#include <string.h>
#include <dirent.h>
#include <fcntl.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/stat.h>

#include <linux/ext2_fs.h>

const char *prog_name = "chattr";

int add = 0;
int rem = 0;
int set = 0;
int set_version  = 0;
unsigned long cluster_size = 0;
unsigned long compr_meth   = -1;

unsigned long version;

int recursive = 0;
int verbose   = 0;

unsigned long af = 0;
unsigned long rf = 0;
unsigned long sf = 0;

/*
 *	Do it for every file in the directory.
 */

static void chattr (char *path, int len);

static void chattr_dir (char *path, int len)
{
  DIR *dir = opendir (path);

  if (dir == NULL)
    return;

  else {
    struct dirent *de;

    path[len++] = '/';
    path[len]   = '\0';

    while ((de = readdir (dir)) != NULL)
      if (de->d_name[0] != '.' || (de->d_name[1] != '\0' && (de->d_name[1] != '.' || de->d_name[2] != '\0')))
	{
	  strcpy (path + len, de->d_name);
	  chattr (path, len + strlen (de->d_name));
	}

    closedir (dir);
  }
}

/*
 *	Do it for a single file, and go deeper if recursive.
 */

static void chattr (char *path, int len)
{
  int fd;
  struct stat st;

  if (lstat (path, &st) == -1)
    fprintf (stderr, "%s: %s: can't stat file (%s)\n",
	     prog_name, path, strerror (errno));

  if (S_ISDIR (st.st_mode) || S_ISREG (st.st_mode)) {
    if ((fd = open (path, O_RDONLY)) == -1)
      fprintf (stderr, "%s: %s: can't open file (%s)\n",
	       prog_name, path, strerror (errno));
    else {
      unsigned long o_flags;
      
      /*
       *	First, get the old flags.
       */

      if (ioctl (fd, EXT2_IOC_GETFLAGS, &o_flags) == -1)
	fprintf (stderr, "%s: %s: can't read flags (%s)\n",
		 prog_name, path, strerror (errno));
      else {
	unsigned long n_flags = o_flags;

	if (set)
	  n_flags = sf;
	else {
	  if (rem)
	    n_flags &= ~rf;
	  if (add)
	    n_flags |=  af;
	}
	  
	if (ioctl (fd, EXT2_IOC_SETFLAGS, &n_flags) == -1)
	  fprintf (stderr, "%s: can't set flags (%s)\n",
		   path, strerror (errno));
      }
  
      if (set_version
	  && (ioctl (fd, EXT2_IOC_SETVERSION, &version) == -1))
	fprintf (stderr, "%s: can't set version (%s)\n",
		 path, strerror (errno));

#ifdef EXT2_COMPRESS
      if (cluster_size
	  && (ioctl (fd, EXT2_IOC_SETCLUSTERSIZE, &cluster_size) == -1))
	fprintf (stderr, "%s: can't set cluster size (%s)\n",
		 path, strerror (errno));
    
      if ((compr_meth != -1)
	  && (ioctl (fd, EXT2_IOC_SETCOMPRMETHOD, &compr_meth) == -1))
	fprintf (stderr, "%s: can't set compression method (%s)\n",
		 path, strerror (errno));
#endif

      close (fd);

      /*
       *	Do subdirectories.
       */

      if (recursive && S_ISDIR (st.st_mode)) 
	chattr_dir (path, len);
    }
  }
}

/*
 *	Display some help.
 */

static void usage (int rc)
{
  fprintf (stderr, 
	   "usage: %s [-ER] [-+=acdisuASX] [-b size] [-m name] [-v version] files ...\n"
#if EXT2_APPEND_FL
	   "  +-=a       set/reset append-only flag\n"
#endif
	   "  +-=c       set/reset compress flag\n"
#if EXT2_NODUMP_FL
	   "  +-=d       set/reset no-dump flag\n"
#endif
#if EXT2_IMMUTABLE_FL
	   "  +-=i       set/reset immutable flag\n"
#endif
	   "  +-=s       set/reset secure deletion flag\n"
	   "  +-=u       set/reset undelete flag\n"
#if EXT2_NOATIME_FL
	   "  +-=A	 set/reset no-atime flag\n"
#endif
	   "  +-=S       set/reset sync flag\n"
#if EXT2_NOCOMPR_FL
	   "  +-=X       enable or disable compression on I/O\n"
#endif
#ifdef EXT2_COMPRESS
	   "  -E         reset compression error flag\n"
	   "  -b size    set cluster size (in blocks)\n"
	   "  -m name    set compression method\n"
#endif
	   "  -v number  set file version\n"
	   "  -R         recursive mode\n"
	   "  -h         show this help message\n"
	   , prog_name);

  exit (rc);
}

/*
 *	Some predefined algorithm names.
 */

#ifdef EXT2_COMPRESS

static struct { char const *name; int id; } name2id[] = {
  { "lzv"	, EXT2_LZV1_ID	 	},
  { "lzv1"	, EXT2_LZV1_ID	 	},
  { "lzrw3a"	, EXT2_LZRW3A_ID	},
  { "gzip"	, EXT2_GZIP6_ID	 	},
  { "gzip1"	, EXT2_GZIP1_ID	 	},
  { "gzip2"	, EXT2_GZIP2_ID	 	},
  { "gzip3"	, EXT2_GZIP3_ID	 	},
  { "gzip4"	, EXT2_GZIP4_ID	 	},
  { "gzip5"	, EXT2_GZIP5_ID	 	},
  { "gzip6"	, EXT2_GZIP6_ID	 	},
  { "gzip7"	, EXT2_GZIP7_ID	 	},
  { "gzip8"	, EXT2_GZIP8_ID	 	},
  { "gzip9"	, EXT2_GZIP9_ID	 	},
  { NULL }
};

#endif

/*
 *	Parse arguments.
 */

static int decode_arg (int *i, int argc, char **argv)
{
  char *p;
  char *tmp;
  int   n = *i;

  switch (argv[n][0]) {

  case '-':
    for (p = &argv[n][1]; *p; p++)
      switch (*p) {

#ifdef EXT2_NOATIME_FL
      case 'A' : rf |= EXT2_NOATIME_FL;   rem = 1; break;
#endif
      case 'E' : rf |= EXT2_ECOMPR_FL;	  rem = 1; break;
      case 'S' : rf |= EXT2_SYNC_FL;	  rem = 1; break;
#ifdef EXT2_NOCOMPR_FL
      case 'X' : rf |= EXT2_NOCOMPR_FL;   rem = 1; break;
#endif
#ifdef	EXT2_APPEND_FL
      case 'a' : rf |= EXT2_APPEND_FL;	  rem = 1; break;
#endif
      case 'c' : rf |= EXT2_COMPR_FL;	  rem = 1; break;
#ifdef	EXT2_NODUMP_FL
      case 'd' : rf |= EXT2_NODUMP_FL;	  rem = 1; break;
#endif
#ifdef	EXT2_IMMUTABLE_FL
      case 'i' : rf |= EXT2_IMMUTABLE_FL; rem = 1; break;
#endif
      case 's' : rf |= EXT2_SECRM_FL;	  rem = 1; break;
      case 'u' : rf |= EXT2_UNRM_FL;	  rem = 1; break;

#ifdef EXT2_COMPRESS
      case 'b' : 
	cluster_size = strtol (argv[++(*i)], &tmp, 0);
	if (*tmp)
	  usage (1);
	if (cluster_size !=  4 && cluster_size !=  8 &&
	    cluster_size != 16 && cluster_size != 32)
	  {
	    fprintf (stderr, "%s: invalid cluster size: %s\n", prog_name, argv[*i]);
	    exit (1);
	  }
	break;

      case 'm' :
	{
	  int x;

	  for (x = 0, ++(*i); name2id[x].name != NULL; x++)
	    if (strcasecmp (argv[*i], name2id[x].name) == 0) break;
	
	  if (name2id[x].name == NULL)
	    { fprintf (stderr, "%s: unknown compression method: %s\n", prog_name, argv[*i]); exit (1); }

	  compr_meth = name2id[x].id;
	}
	break;
#endif
      case 'v' :
	version = strtol (argv[++(*i)], &tmp, 0);
	if (*tmp)
	  {
	    fprintf (stderr, "%s: bad version: %s\n", prog_name, argv[*i]);
	    exit (1);
	  }
	set_version = 1;
	break;

      case 'R' : recursive = 1;	break;
      case 'V' : verbose   = 1;	break; /* Unimplemented */
      case 'h' : usage (0);

      default: usage (1);
      }
    break;

  case '+':
    add = 1;

    for (p = &argv[n][1]; *p; p++)
      switch (*p) {
#ifdef EXT2_NOATIME_FL
        case 'A' : af |= EXT2_NOATIME_FL;   break;
#endif
	case 'S' : af |= EXT2_SYNC_FL;	    break;
#ifdef EXT2_NOCOMPR_FL
	case 'X' : af |= EXT2_NOCOMPR_FL;   break;
#endif
#ifdef	EXT2_APPEND_FL
	case 'a' : af |= EXT2_APPEND_FL;    break;
#endif
	case 'c' : af |= EXT2_COMPR_FL;	    break;
#ifdef	EXT2_NODUMP_FL
	case 'd' : af |= EXT2_NODUMP_FL;    break;
#endif
#ifdef	EXT2_IMMUTABLE_FL
	case 'i' : af |= EXT2_IMMUTABLE_FL; break;
#endif
	case 's' : af |= EXT2_SECRM_FL;	    break;
	case 'u' : af |= EXT2_UNRM_FL;	    break;

	default  : usage (1);
	}
    break;

  case '=' :
    set = 1;
	
    for (p = &argv[n][1]; *p; p++)
      switch (*p) {
#ifdef EXT2_NOATIME_FL
      case 'A' : sf |= EXT2_NOATIME_FL;   break;
#endif
      case 'S' : sf |= EXT2_SYNC_FL;	  break;
#ifdef EXT2_NOCOMPR_FL
      case 'X' : sf |= EXT2_NOCOMPR_FL;	  break;
#endif
#ifdef	EXT2_APPEND_FL
      case 'a' : sf |= EXT2_APPEND_FL;	  break;
#endif
      case 'c' : sf |= EXT2_COMPR_FL;	  break;
#ifdef	EXT2_NODUMP_FL
      case 'd' : sf |= EXT2_NODUMP_FL;	  break;
#endif
#ifdef	EXT2_IMMUTABLE_FL
      case 'i' : sf |= EXT2_IMMUTABLE_FL; break;
#endif
      case 's' : sf |= EXT2_SECRM_FL;	  break;
      case 'u' : sf |= EXT2_UNRM_FL;	  break;

      default  : usage (1);
      }
    break;

  default : return EOF;
  }
	
  return 1;
}

int main (int argc, char *argv[])
{
  int i;
  char path[MAXPATHLEN];

  if (argc && *argv)
    prog_name = *argv;

  for (i = 1; i < argc; i++)
    if (decode_arg (&i, argc, argv) == EOF)
      break;

  if (i >= argc)
    usage (1);

  if (set && (add || rem))
    {
      fprintf (stderr, "%s: = is incompatible with - and +\n", prog_name);
      exit (1);
    }

  if ((rf & af) != 0)
    {
      fprintf (stderr, "Can't both set and unset same flag.\n");
      exit (1);
    }

  while (i < argc)
    {
      strcpy (path, argv[i]);
      chattr (path, strlen (path));
      i++;
    }

  return 0;
}
